/*
    SPDX-FileCopyrightText: 2009 Milian Wolff <mail@milianw.de>

    SPDX-License-Identifier: LGPL-2.0-or-later
*/

#include "colorcache.h"

#include "configurablecolors.h"
#include "codehighlighting.h"

#include <KColorScheme>

#include "../../interfaces/icore.h"
#include "../../interfaces/ilanguagecontroller.h"
#include "../../interfaces/icompletionsettings.h"
#include "../../interfaces/idocument.h"
#include "../../interfaces/idocumentcontroller.h"
#include "../interfaces/ilanguagesupport.h"
#include "../duchain/duchain.h"
#include "../duchain/duchainlock.h"
#include <debug.h>
#include "widgetcolorizer.h"

#include <KTextEditor/Document>
#include <KTextEditor/View>
#include <KSyntaxHighlighting/Definition>
#include <KSyntaxHighlighting/Format>

#define ifDebug(x)

namespace KDevelop {
ColorCache* ColorCache::m_self = nullptr;

CodeHighlightingType highlightingTypeFromName(const QString& name)
{
    if (name == QLatin1String("Class")) {
        return CodeHighlightingType::Class;
    } else if (name == QLatin1String("Local Member Variable")) {
        return CodeHighlightingType::LocalClassMember;
    } else if (name == QLatin1String("Local Member Function")) {
        return CodeHighlightingType::LocalMemberFunction;
    } else if (name == QLatin1String("Inherited Member Variable")) {
        return CodeHighlightingType::InheritedClassMember;
    } else if (name == QLatin1String("Inherited Member Function")) {
        return CodeHighlightingType::InheritedMemberFunction;
    } else if (name == QLatin1String("Function")) {
        return CodeHighlightingType::Function;
    } else if (name == QLatin1String("Function Argument")) {
        return CodeHighlightingType::FunctionVariable;
    } else if (name == QLatin1String("Type Alias")) {
        return CodeHighlightingType::TypeAlias;
    } else if (name == QLatin1String("Forward Declaration")) {
        return CodeHighlightingType::ForwardDeclaration;
    } else if (name == QLatin1String("Namespace")) {
        return CodeHighlightingType::Namespace;
    } else if (name == QLatin1String("Local Variable")) {
        return CodeHighlightingType::LocalVariable;
    } else if (name == QLatin1String("Global Variable")) {
        return CodeHighlightingType::GlobalVariable;
    } else if (name == QLatin1String("Member Variable")) {
        return CodeHighlightingType::MemberVariable;
    } else if (name == QLatin1String("Namespace Variable")) {
        return CodeHighlightingType::NamespaceVariable;
    } else if (name == QLatin1String("Enumeration")) {
        return CodeHighlightingType::Enum;
    } else if (name == QLatin1String("Enumerator")) {
        return CodeHighlightingType::Enumerator;
    } else if (name == QLatin1String("Macro")) {
        return CodeHighlightingType::Macro;
    } else if (name == QLatin1String("Macro Function")) {
        return CodeHighlightingType::MacroFunctionLike;
    }
    return CodeHighlightingType::Error;
}

ColorCache::ColorCache(QObject* parent)
    : QObject(parent)
    , m_defaultColors(new ConfigurableHighlightingColors)
    , m_validColorCount(0)
    , m_localColorRatio(0)
    , m_globalColorRatio(0)
    , m_globalColorSource(ICompletionSettings::GlobalColorSource::AutoGenerated)
    , m_boldDeclarations(true)
{
    Q_ASSERT(m_self == nullptr);

    updateColorsFromScheme(); // default / fallback
    updateColorsFromSettings();

    connect(ICore::self()->languageController()->completionSettings(), &ICompletionSettings::settingsChanged,
            this, &ColorCache::updateColorsFromSettings, Qt::QueuedConnection);

    connect(ICore::self()->documentController(), &IDocumentController::documentActivated,
            this, &ColorCache::slotDocumentActivated);

    bool hadDoc = tryActiveDocument();

    updateInternal();

    m_self = this;

    if (!hadDoc) {
        // try to update later on again
        QMetaObject::invokeMethod(this, "tryActiveDocument", Qt::QueuedConnection);
    }
}

bool ColorCache::tryActiveDocument()
{
    KTextEditor::View* view = ICore::self()->documentController()->activeTextDocumentView();
    if (view) {
        updateColorsFromView(view);
        return true;
    }
    return false;
}

ColorCache::~ColorCache()
{
    m_self = nullptr;
    delete m_defaultColors;
    m_defaultColors = nullptr;
}

ColorCache* ColorCache::self()
{
    if (!m_self) {
        m_self = new ColorCache;
    }
    return m_self;
}

void ColorCache::generateColors()
{
    // Primary colors taken from: http://colorbrewer2.org/?type=qualitative&scheme=Paired&n=12
    const QColor colors[] = {{0xb15928}, {0xff7f00}, {0xb2df8a}, {0x33a02c}, {0xa6cee3},
                             {0x1f78b4}, {0x6a3d9a}, {0xcab2d6}, {0xe31a1c}, {0xfb9a99}};
    const int colorCount = std::extent<decltype(colors)>::value;

    // Supplementary colors generated by: http://tools.medialab.sciences-po.fr/iwanthue/
    const QColor supplementaryColors[] = {{0xD33B67}, {0x5EC764}, {0x6CC82D}, {0x995729}, {0xFB4D84}, {0x4B8828},
                                          {0xD847D0}, {0xB56AC5}, {0xE96F0C}, {0xDC7161}, {0x4D7279}, {0x01AAF1},
                                          {0xD2A237}, {0xF08CA5}, {0xC83E93}, {0x5D7DF7}, {0xEFBB51}, {0x108BBB},
                                          {0x5C84B8}, {0x02F8BC}, {0xA5A9F7}, {0xF28E64}, {0xA461E6}, {0x6372D3}};
    const int supplementaryColorCount = std::extent<decltype(supplementaryColors)>::value;

    m_colors.clear();
    m_colors.reserve(colorCount + supplementaryColorCount);

    for (const auto& color: colors) {
        m_colors.append(blendLocalColor(color));
    }

    m_primaryColorCount = m_colors.count();

    for (const auto& color: supplementaryColors) {
        m_colors.append(blendLocalColor(color));
    }

    m_validColorCount = m_colors.count();
}

void ColorCache::slotDocumentActivated()
{
    KTextEditor::View* view = ICore::self()->documentController()->activeTextDocumentView();
    ifDebug(qCDebug(LANGUAGE) << "doc activated:" << doc; )
    if (view) {
        updateColorsFromView(view);
    }
}

void ColorCache::slotViewSettingsChanged()
{
    auto* view = qobject_cast<KTextEditor::View*>(sender());
    Q_ASSERT(view);

    ifDebug(qCDebug(LANGUAGE) << "settings changed" << view; )
    updateColorsFromView(view);
}

void ColorCache::updateColorsFromView(KTextEditor::View* view)
{
    if (!view) {
        // yeah, the HighlightInterface methods returning an Attribute
        // require a View... kill me for that mess
        return;
    }

    QColor foreground(QColor::Invalid);
    QColor background(QColor::Invalid);

    const auto style = view->defaultStyleAttribute(KSyntaxHighlighting::Theme::TextStyle::Normal);
    foreground = style->foreground().color();
    if (style->hasProperty(QTextFormat::BackgroundBrush)) {
        background = style->background().color();
    }

    if (KTextEditor::View* view = m_view.data()) {
        // we only listen to a single view, i.e. the active one
        disconnect(view, &KTextEditor::View::configChanged, this, &ColorCache::slotViewSettingsChanged);
    }
    connect(view, &KTextEditor::View::configChanged, this, &ColorCache::slotViewSettingsChanged);
    m_view = view;

    bool anyAttrChanged = false;
    if (!foreground.isValid()) {
        // fallback to colorscheme variant
        ifDebug(qCDebug(LANGUAGE) << "updating from scheme"; )
        updateColorsFromScheme();
    } else if (m_foregroundColor != foreground || m_backgroundColor != background) {
        m_foregroundColor = foreground;
        m_backgroundColor = background;
        m_defaultColors->reset(this, view);
        anyAttrChanged = true;
    }

    anyAttrChanged |= updateColorsFromTheme(view->theme());

    if (anyAttrChanged) {
        ifDebug(qCDebug(LANGUAGE) << "updating from document"; )
        update();
    }
}

bool ColorCache::updateColorsFromTheme(const KSyntaxHighlighting::Theme& theme)
{
    if (m_globalColorSource != ICompletionSettings::GlobalColorSource::FromTheme)
        return false;

    // from ktexteditor/src/syntax/kateextendedattribute.h
    static const int SelectedBackground = QTextFormat::UserProperty + 2;

    const auto schemeDefinition = m_schemeRepo.definitionForName(QStringLiteral("Semantic Colors"));
    const auto formats = schemeDefinition.formats();

    bool anyAttrChanged = false;
    for (const auto& format : formats) {
        const auto type = highlightingTypeFromName(format.name());
        if (type == CodeHighlightingType::Error) {
            continue;
        }
        const auto attr = m_defaultColors->attribute(type);
        Q_ASSERT(attr);

        auto forwardProperty = [&](auto formatGetter, auto attrProperty, auto attrSetter) {
            auto formatProperty = (format.*formatGetter)(theme);
            if (attrProperty != formatProperty) {
                (attr.data()->*attrSetter)(formatProperty);
                anyAttrChanged = true;
            }
        };

        using namespace KSyntaxHighlighting;
        using namespace KTextEditor;
        forwardProperty(&Format::isBold, attr->fontBold(), &Attribute::setFontBold);
        forwardProperty(&Format::isItalic, attr->fontItalic(), &Attribute::setFontItalic);
        forwardProperty(&Format::isUnderline, attr->fontUnderline(), &Attribute::setFontUnderline);
        forwardProperty(&Format::isStrikeThrough, attr->fontStrikeOut(), &Attribute::setFontStrikeOut);
        forwardProperty(&Format::textColor, attr->foreground().color(), &Attribute::setForeground);
        forwardProperty(&Format::selectedTextColor, attr->selectedForeground().color(),
                        &Attribute::setSelectedForeground);

        if (format.hasBackgroundColor(theme)) {
            forwardProperty(&Format::backgroundColor, attr->background().color(), &Attribute::setBackground);
        } else if (type == CodeHighlightingType::HighlightUses) {
            auto background = QColor::fromRgb(theme.editorColor(KSyntaxHighlighting::Theme::SearchHighlight));
            if (attr->background().color() != background) {
                attr->setBackground(background);
                anyAttrChanged = true;
            }
        } else if (attr->background() != QBrush()) {
            attr->setBackground(QBrush());
            anyAttrChanged = true;
        }

        // from KSyntaxHighlighting::Format::isDefaultTextStyle
        if (format.selectedBackgroundColor(theme).rgba() != theme.selectedBackgroundColor(KSyntaxHighlighting::Theme::Normal)) {
            forwardProperty(&Format::selectedBackgroundColor, attr->selectedBackground().color(),
                            &Attribute::setSelectedBackground);
        } else if (attr->hasProperty(SelectedBackground)) {
            attr->clearProperty(SelectedBackground);
            anyAttrChanged = true;
        }
    }
    return anyAttrChanged;
}

void ColorCache::updateColorsFromScheme()
{
    KColorScheme scheme(QPalette::Normal, KColorScheme::View);

    QColor foreground = scheme.foreground(KColorScheme::NormalText).color();
    QColor background = scheme.background(KColorScheme::NormalBackground).color();

    if (foreground != m_foregroundColor || background != m_backgroundColor) {
        m_foregroundColor = foreground;
        m_backgroundColor = background;
        update();
    }
}

void ColorCache::updateColorsFromSettings()
{
    auto settings = ICore::self()->languageController()->completionSettings();

    const auto globalColorSource = settings->globalColorSource();
    const auto globalColorSourceChanged = globalColorSource != m_globalColorSource;
    m_globalColorSource = globalColorSource;

    const auto globalRatio = settings->globalColorizationLevel();
    const auto globalRatioChanged = globalRatio != m_globalColorRatio;
    m_globalColorRatio = globalRatio;

    const auto localRatio = settings->localColorizationLevel();
    const auto localRatioChanged = localRatio != m_localColorRatio;
    m_localColorRatio = localRatio;

    const auto boldDeclartions = settings->boldDeclarations();
    const auto boldDeclarationsChanged = boldDeclartions != m_boldDeclarations;
    m_boldDeclarations = boldDeclartions;

    if (m_view && (globalRatioChanged || globalColorSourceChanged)) {
        updateDefaultColorsFromSource();
    }

    if (globalColorSourceChanged || globalRatioChanged || localRatioChanged || boldDeclarationsChanged) {
        update();
    }
}

void ColorCache::updateDefaultColorsFromSource()
{
    switch (m_globalColorSource) {
    case ICompletionSettings::GlobalColorSource::AutoGenerated:
        m_defaultColors->reset(this, m_view.data());
        break;
    case ICompletionSettings::GlobalColorSource::FromTheme:
        updateColorsFromTheme(m_view->theme());
        break;
    }
}

void ColorCache::update()
{
    if (!m_self) {
        ifDebug(qCDebug(LANGUAGE) << "not updating - still initializating"; )
        // don't update on startup, updateInternal is called directly there
        return;
    }

    QMetaObject::invokeMethod(this, "updateInternal", Qt::QueuedConnection);
}

void ColorCache::updateInternal()
{
    ifDebug(qCDebug(LANGUAGE) << "update internal" << m_self; )
    generateColors();

    if (!m_self) {
        // don't do anything else fancy on startup
        return;
    }

    emit colorsGotChanged();

    // rehighlight open documents
    if (!ICore::self() || ICore::self()->shuttingDown()) {
        return;
    }
    const auto documents = ICore::self()->documentController()->openDocuments();
    for (IDocument* doc : documents) {
        const auto languages = ICore::self()->languageController()->languagesForUrl(doc->url());
        for (const auto lang : languages) {
            ReferencedTopDUContext top;
            {
                DUChainReadLocker lock;
                top = lang->standardContext(doc->url());
            }

            if (top) {
                if (ICodeHighlighting* highlighting = lang->codeHighlighting()) {
                    highlighting->highlightDUChain(top);
                }
            }
        }
    }
}

QColor ColorCache::blend(QColor color, uchar ratio) const
{
    Q_ASSERT(m_backgroundColor.isValid());
    Q_ASSERT(m_foregroundColor.isValid());
    return WidgetColorizer::blendForeground(color, float( ratio ) / float( 0xff ), m_foregroundColor,
                                            m_backgroundColor);
}

QColor ColorCache::blendBackground(QColor color, uchar ratio) const
{
    return WidgetColorizer::blendBackground(color, float( ratio ) / float( 0xff ), m_foregroundColor,
                                            m_backgroundColor);
}

QColor ColorCache::blendGlobalColor(QColor color) const
{
    return blend(color, m_globalColorRatio);
}

QColor ColorCache::blendLocalColor(QColor color) const
{
    return blend(color, m_localColorRatio);
}

ConfigurableHighlightingColors* ColorCache::defaultColors() const
{
    return m_defaultColors;
}

QColor ColorCache::generatedColor(uint num) const
{
    return num > ( uint )m_colors.size() ? foregroundColor() : m_colors[num];
}

uint ColorCache::validColorCount() const
{
    return m_validColorCount;
}

uint ColorCache::primaryColorCount() const
{
    return m_primaryColorCount;
}

QColor ColorCache::foregroundColor() const
{
    return m_foregroundColor;
}
}

#include "moc_colorcache.cpp"
