package internal

import "sync"

// List indexes instances by ID and maintains insertion order for quick access
// to the oldest instances.
type List struct {
	mu    sync.Mutex
	index map[string]*Instance
	list  []*Instance
}

func (a *List) Get(id string) *Instance {
	a.mu.Lock()
	defer a.mu.Unlock()

	if a.index == nil {
		return nil
	}
	return a.index[id]
}

func (a *List) Add(inst *Instance) {
	a.mu.Lock()
	defer a.mu.Unlock()

	if a.index == nil {
		a.index = make(map[string]*Instance)
	}

	a.index[inst.ID()] = inst
	a.list = append(a.list, inst)
}

func (a *List) Delete(id string) {
	a.mu.Lock()
	defer a.mu.Unlock()

	delete(a.index, id)
	for idx := range a.list {
		if a.list[idx].ID() == id {
			a.list = append(a.list[:idx], a.list[idx+1:]...)
			return
		}
	}
}

type Slots struct {
	Idle        int // idle is slots available for tasks
	Acquired    int // acquired is slots that already have a task assigned
	Unavailable int // unavailable is slots that cannot be assigned (due to use count)
	Unhealthy   int // unhealthy is slots that cannot be assigned because the instance backing the capacity is unhealthy
	Unready     int // unready is slots that are unready because the instance is not ready
}

// Slots returns the idle, acquired, unavailable and unhealthy capacity as
// a total of all available instances.
func (a *List) Slots() Slots {
	a.mu.Lock()
	defer a.mu.Unlock()

	var s Slots
	for _, instance := range a.list {
		x, y, deleted := instance.Capacity()
		if deleted {
			continue
		}

		s.Acquired += x
		s.Unavailable += y

		capacity := instance.state.CapacityPerInstance - x - y
		if instance.IsReady() {
			switch {
			case instance.Unhealthy():
				s.Unhealthy += capacity
			default:
				s.Idle += capacity
			}
		} else {
			s.Unready += capacity
		}
	}

	return s
}

func (a *List) List() []*Instance {
	a.mu.Lock()
	defer a.mu.Unlock()

	list := make([]*Instance, len(a.list))
	copy(list, a.list)

	return list
}
