package taskscaler

import (
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"gitlab.com/gitlab-org/fleeting/fleeting"
)

func TestNewOptions(t *testing.T) {

	tests := map[string]struct {
		// config values
		configMaxInstances                int
		configCapacityPerInstance         int
		configMaxUseCount                 int
		configFailureThreshold            int
		configAcquireDelay                time.Duration
		configScaleThrottleLimit          int
		configScaleThrottleBurst          int
		configUpdateInterval              time.Duration
		configUpdateIntervalWhenExpecting time.Duration
		configDeletionRetryInterval       time.Duration
		configShutdownDeletionRetries     int
		configShutdownDeletionInterval    time.Duration
		// expected option values
		expectedMaxInstances                int
		expectedCapacityPerInstance         int
		expectedMaxUseCount                 int
		expectedFailureThreshold            int
		expectedAcquireDelay                time.Duration
		expectedScaleThrottleLimit          int
		expectedScaleThrottleBurst          int
		expectedUpdateInterval              time.Duration
		expectedUpdateIntervalWhenExpecting time.Duration
		expectedDeletionRetryInterval       time.Duration
		expectedShutdownDeletionRetries     int
		expectedShutdownDeletionInterval    time.Duration
	}{
		"majority of config values set to zero": {
			configMaxInstances:                1,
			configCapacityPerInstance:         0,
			configMaxUseCount:                 0,
			configFailureThreshold:            0,
			configAcquireDelay:                1 * time.Second,
			configScaleThrottleLimit:          0,
			configScaleThrottleBurst:          0,
			configUpdateInterval:              0,
			configUpdateIntervalWhenExpecting: 0,
			configDeletionRetryInterval:       0,
			configShutdownDeletionRetries:     0,
			configShutdownDeletionInterval:    0,

			expectedMaxInstances:                1, // expect the config value
			expectedCapacityPerInstance:         DefaultCapacityPerInstance,
			expectedMaxUseCount:                 0, // expect the config value
			expectedFailureThreshold:            DefaultFailureThreshold,
			expectedAcquireDelay:                1 * time.Second,
			expectedScaleThrottleLimit:          DefaultScaleThrottleLimit,
			expectedScaleThrottleBurst:          1, // we expect the burst value to be 1 - max instances
			expectedUpdateInterval:              fleeting.DefaultUpdateInterval,
			expectedUpdateIntervalWhenExpecting: fleeting.DefaultUpdateIntervalWhenExpecting,
			expectedDeletionRetryInterval:       fleeting.DefaultDeletionRetryInterval,
			expectedShutdownDeletionRetries:     fleeting.DefaultShutdownDeletionRetries,
			expectedShutdownDeletionInterval:    fleeting.DefaultShutdownDeletionInterval,
		},
		"majority of config values set to defaults": {
			configMaxInstances:                10,
			configCapacityPerInstance:         DefaultCapacityPerInstance,
			configMaxUseCount:                 10,
			configFailureThreshold:            DefaultFailureThreshold,
			configAcquireDelay:                DefaultAcquireDelay,
			configScaleThrottleLimit:          DefaultScaleThrottleLimit,
			configScaleThrottleBurst:          5,
			configUpdateInterval:              fleeting.DefaultUpdateInterval,
			configUpdateIntervalWhenExpecting: fleeting.DefaultUpdateIntervalWhenExpecting,
			configDeletionRetryInterval:       fleeting.DefaultDeletionRetryInterval,
			configShutdownDeletionRetries:     fleeting.DefaultShutdownDeletionRetries,
			configShutdownDeletionInterval:    fleeting.DefaultShutdownDeletionInterval,

			expectedMaxInstances:                10,
			expectedCapacityPerInstance:         DefaultCapacityPerInstance,
			expectedMaxUseCount:                 10,
			expectedFailureThreshold:            DefaultFailureThreshold,
			expectedAcquireDelay:                DefaultAcquireDelay,
			expectedScaleThrottleLimit:          DefaultScaleThrottleLimit,
			expectedScaleThrottleBurst:          5, // we expect the burst value to be 5 - the config value
			expectedUpdateInterval:              fleeting.DefaultUpdateInterval,
			expectedUpdateIntervalWhenExpecting: fleeting.DefaultUpdateIntervalWhenExpecting,
			expectedDeletionRetryInterval:       fleeting.DefaultDeletionRetryInterval,
			expectedShutdownDeletionRetries:     fleeting.DefaultShutdownDeletionRetries,
			expectedShutdownDeletionInterval:    fleeting.DefaultShutdownDeletionInterval,
		},
		"some custom config with values != 0": {
			configMaxInstances:                50,
			configCapacityPerInstance:         5,
			configMaxUseCount:                 50,
			configFailureThreshold:            2,
			configAcquireDelay:                3 * time.Second,
			configScaleThrottleLimit:          50,
			configScaleThrottleBurst:          0, // we want to see that max instance value is used
			configUpdateInterval:              30 * time.Second,
			configUpdateIntervalWhenExpecting: 1 * time.Second,
			configDeletionRetryInterval:       90 * time.Second,
			configShutdownDeletionRetries:     5,
			configShutdownDeletionInterval:    15 * time.Second,

			expectedMaxInstances:                50,
			expectedCapacityPerInstance:         5,
			expectedMaxUseCount:                 50,
			expectedFailureThreshold:            2,
			expectedAcquireDelay:                3 * time.Second,
			expectedScaleThrottleLimit:          50,
			expectedScaleThrottleBurst:          50, // we expect the burst value to be 50 - max instances value
			expectedUpdateInterval:              30 * time.Second,
			expectedUpdateIntervalWhenExpecting: 1 * time.Second,
			expectedDeletionRetryInterval:       90 * time.Second,
			expectedShutdownDeletionRetries:     5,
			expectedShutdownDeletionInterval:    15 * time.Second,
		},
	}

	for tn, tc := range tests {
		t.Run(tn, func(t *testing.T) {
			options := []Option{
				WithCapacityPerInstance(tc.configCapacityPerInstance),
				WithMaxUseCount(tc.configMaxUseCount),
				WithMaxInstances(tc.configMaxInstances),
				WithUpdateInterval(tc.configUpdateInterval),
				WithUpdateIntervalWhenExpecting(tc.configUpdateIntervalWhenExpecting),
				WithDeletionRetryInterval(tc.configDeletionRetryInterval),
				WithShutdownDeletionInterval(tc.configShutdownDeletionInterval),
				WithShutdownDeletionRetries(tc.configShutdownDeletionRetries),
				WithFailureThreshold(tc.configFailureThreshold),
				WithScaleThrottle(tc.configScaleThrottleLimit, tc.configScaleThrottleBurst),
				WithAcquireDelay(tc.configAcquireDelay),
			}

			opts, err := newOptions(options)

			assert.NoError(t, err)

			assert.Equal(t, tc.expectedMaxInstances, opts.maxInstances, "max instances is not %d", tc.expectedMaxInstances)
			assert.Equal(t, tc.expectedCapacityPerInstance, opts.capacityPerInstance, "capacity per instance is not %d", tc.expectedCapacityPerInstance)
			assert.Equal(t, tc.expectedMaxUseCount, opts.maxUseCount, "max use count is not %d", tc.expectedMaxUseCount)
			assert.Equal(t, tc.expectedFailureThreshold, opts.failureThreshold, "update interval is not %d", tc.expectedFailureThreshold)
			assert.Equal(t, tc.expectedAcquireDelay, opts.acquireDelay, "acquire delay is not %d", tc.expectedAcquireDelay)
			assert.Equal(t, tc.expectedScaleThrottleLimit, opts.scaleThrottleLimit, "scale throttle limit is not %d", tc.expectedScaleThrottleLimit)
			assert.Equal(t, tc.expectedScaleThrottleBurst, opts.scaleThrottleBurst, "scale throttle burst is not %d", tc.expectedScaleThrottleBurst)
			assert.Equal(t, tc.expectedUpdateInterval, opts.updateInterval, "update interval is not %d", tc.expectedUpdateInterval)
			assert.Equal(t, tc.expectedUpdateIntervalWhenExpecting, opts.updateIntervalWhenExpecting, "update interval when expecting is not %d", tc.expectedUpdateIntervalWhenExpecting)
			assert.Equal(t, tc.expectedDeletionRetryInterval, opts.deletionRetryInterval, "deletion retry interval is not %d", tc.expectedDeletionRetryInterval)
			assert.Equal(t, tc.expectedShutdownDeletionInterval, opts.shutdownDeletionInterval, "shutdown deletion interval is not %d", tc.expectedShutdownDeletionInterval)
			assert.Equal(t, tc.expectedShutdownDeletionRetries, opts.shutdownDeletionRetries, "shutdown deletion retries is not %d", tc.expectedShutdownDeletionRetries)
		})
	}
}
