package log

import (
	"bufio"
	"net/http"
	"net/url"
	"strings"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestComputeApproximateRequestSize(t *testing.T) {
	tests := []struct {
		name     string
		request  func() *http.Request
		expected int64
	}{
		{
			name: "empty request",
			request: func() *http.Request {
				return &http.Request{
					Method: http.MethodGet,
					URL:    &url.URL{Path: "/"},
					Proto:  "HTTP/1.1",
					Header: http.Header{},
					Host:   "",
				}
			},
			expected: int64(18),
		},
		{
			name: "request with nil URL",
			request: func() *http.Request {
				return &http.Request{
					Method: http.MethodGet,
					URL:    nil,
					Proto:  "HTTP/1.1",
					Header: http.Header{},
					Host:   "example.com",
				}
			},
			expected: int64(35),
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			req := tt.request()
			size := computeApproximateRequestSize(req, 0)
			require.Equal(t, tt.expected, size, "computeApproximateRequestSize() returned unexpected size")
		})
	}
}

func TestComputeApproximateRequestSizeRaw(t *testing.T) {
	tests := []struct {
		name          string
		request       string
		bodyByteCount int64
	}{
		{
			name: "basic request",
			request: `
GET / HTTP/1.1

`,
		},
		{
			name: "request with headers",
			request: `
POST /api/v1/users HTTP/1.1
Content-Type: application/json
Accept: application/json
X-Request-ID: abc123
Content-Length: 10
Accept-Encoding: gzip, deflate

0123456789`,
			bodyByteCount: 10,
		},
		{
			name: "request with query parameters",
			request: `
GET /search?q=test&page=1 HTTP/1.1
Host: search.example.com
Content-Type: application/json
Accept: application/json
X-Request-ID: abc123
Accept-Encoding: gzip, deflate

`,
			bodyByteCount: 0,
		},
		{
			name: "request with multiple header values",
			request: `
GET / HTTP/1.1
Accept: application/json
Accept: application/xhtml+xml
Accept: application/xml;q=0.9

`,
			bodyByteCount: 0,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			// Cleanup the request
			cleanedRequest := strings.TrimPrefix(tt.request, "\n")
			cleanedRequest = strings.ReplaceAll(cleanedRequest, "\n", "\r\n")

			rawRequest := strings.NewReader(cleanedRequest)
			req, err := http.ReadRequest(bufio.NewReader(rawRequest))
			require.NoError(t, err, "Failed to create request")

			size := computeApproximateRequestSize(req, tt.bodyByteCount)
			require.Equal(t, int64(len(cleanedRequest)), size, "computeApproximateRequestSize() returned unexpected size")
		})
	}
}
