package expression_test

import (
	"strconv"
	"testing"

	"github.com/stretchr/testify/require"
	"google.golang.org/protobuf/types/known/structpb"

	"gitlab.com/gitlab-org/step-runner/pkg/internal/expression"
	"gitlab.com/gitlab-org/step-runner/pkg/testutil/bldr"
)

func textContextSteps(t *testing.T) *expression.InterpolationContext {
	jobVars := bldr.JobVars(t).WithValue("job_id", "1982").Build()

	stepsCtx := bldr.StepsContext(t).
		WithGlobalContext(bldr.GlobalContext(t).WithJobVars(jobVars).Build()).
		WithEnv("MOVIE", "tron").
		WithEnv("WHERE", "inside").
		WithInput("name", structpb.NewStringValue("Kevin Flynn")).
		WithInput("light_cycle", structpb.NewStructValue(&structpb.Struct{Fields: map[string]*structpb.Value{
			"color":  structpb.NewStringValue("yellow"),
			"number": structpb.NewNumberValue(3),
		}})).
		WithInput("team_members", structpb.NewListValue(&structpb.ListValue{Values: []*structpb.Value{
			structpb.NewStringValue("tron"),
			structpb.NewStringValue("ram"),
			structpb.NewStringValue("flynn"),
		}})).
		Build()

	return stepsCtx.View()
}

func TestExpandString(t *testing.T) {
	expandStringTests := []struct {
		value       string
		expectedStr string
		expectedErr string
	}{{
		value:       "${{job.job_id}}",
		expectedStr: "1982",
	}, {
		value:       "${{ job.job_id }}",
		expectedStr: "1982",
	}, {
		value:       "${{ job.job_id }}:${{ env.MOVIE }}",
		expectedStr: "1982:tron",
	}, {
		value:       "${{ job.undefined_key }}",
		expectedErr: `job.undefined_key: the "undefined_key" was not found`,
	}, {
		value:       `${{ job["${{ job.key }}"] }}`,
		expectedErr: `job["${{ job.key }}"]: the "job[\"${{ job" was not found`,
	}, {
		value:       `${{ job.job_id`,
		expectedErr: `the " job.job_id" is not closed: ${{ ... }}`,
	}, {
		value:       `${{ job.job_id }} }}`,
		expectedErr: `the " job.job_id }} " has extra '}}'`,
	}, {
		value:       `${{inputs.light_cycle}}`,
		expectedStr: `{"color":"yellow","number":3}`,
	}, {
		value:       `PREFIX ${{inputs.light_cycle}} SUFFIX`,
		expectedStr: `PREFIX {"color":"yellow","number":3} SUFFIX`,
	}}
	for _, est := range expandStringTests {
		t.Run(est.value, func(t *testing.T) {
			got, err := expression.ExpandString(textContextSteps(t), est.value)
			if est.expectedErr != "" {
				require.Error(t, err)
				require.ErrorContains(t, err, est.expectedErr)
			} else {
				require.NoError(t, err)
				require.Equal(t, est.expectedStr, got)
			}
		})
	}
}

func TestExpand(t *testing.T) {
	cases := []struct {
		value   *structpb.Value
		want    *structpb.Value
		wantErr error
	}{{
		value: structpb.NewStringValue(""),
		want:  structpb.NewStringValue(""),
	}, {
		value: structpb.NewStringValue("${{job.job_id}}"),
		want:  structpb.NewStringValue("1982"),
	}, {
		value: structpb.NewStringValue("${{env.MOVIE}}"),
		want:  structpb.NewStringValue("tron"),
	}, {
		value: structpb.NewStringValue("${{env.WHERE}}"),
		want:  structpb.NewStringValue("inside"),
	}, {
		value: structpb.NewStringValue("${{inputs.name}}"),
		want:  structpb.NewStringValue("Kevin Flynn"),
	}, {
		value: structpb.NewStringValue("${{inputs.light_cycle}}"),
		want: structpb.NewStructValue(&structpb.Struct{Fields: map[string]*structpb.Value{
			"color":  structpb.NewStringValue("yellow"),
			"number": structpb.NewNumberValue(3),
		}}),
	}, {
		value: structpb.NewStringValue("PREFIX ${{inputs.light_cycle}} SUFFIX"),
		want:  structpb.NewStringValue(`PREFIX {"color":"yellow","number":3} SUFFIX`),
	}, {
		value: structpb.NewStringValue("${{inputs.team_members}}"),
		want: structpb.NewListValue(&structpb.ListValue{Values: []*structpb.Value{
			structpb.NewStringValue("tron"),
			structpb.NewStringValue("ram"),
			structpb.NewStringValue("flynn"),
		}}),
	}, {
		value: structpb.NewStructValue(&structpb.Struct{Fields: map[string]*structpb.Value{
			"replace within": structpb.NewStringValue("${{inputs.name}}"),
		}}),
		want: structpb.NewStructValue(&structpb.Struct{Fields: map[string]*structpb.Value{
			"replace within": structpb.NewStringValue("Kevin Flynn"),
		}}),
	}, {
		value: structpb.NewStructValue(&structpb.Struct{Fields: map[string]*structpb.Value{
			"${{inputs.name}}": structpb.NewStructValue(&structpb.Struct{Fields: map[string]*structpb.Value{
				"${{env.MOVIE}}": structpb.NewStringValue("${{env.WHERE}}"),
			}}),
		}}),
		want: structpb.NewStructValue(&structpb.Struct{Fields: map[string]*structpb.Value{
			"Kevin Flynn": structpb.NewStructValue(&structpb.Struct{Fields: map[string]*structpb.Value{
				"tron": structpb.NewStringValue("inside"),
			}}),
		}}),
	}, {
		value: structpb.NewListValue(&structpb.ListValue{Values: []*structpb.Value{
			structpb.NewStringValue("${{inputs.name}}"),
		}}),
		want: structpb.NewListValue(&structpb.ListValue{Values: []*structpb.Value{
			structpb.NewStringValue("Kevin Flynn"),
		}}),
	}, {
		value: structpb.NewStringValue("Hello, my name is ${{inputs.name}}. You killed my process. Prepare to SIGTERM."),
		want:  structpb.NewStringValue("Hello, my name is Kevin Flynn. You killed my process. Prepare to SIGTERM."),
	}}

	for i, c := range cases {
		t.Run(strconv.Itoa(i), func(t *testing.T) {
			got, err := expression.Expand(textContextSteps(t), c.value)
			if c.wantErr != nil {
				require.Equal(t, c.wantErr, err)
			} else {
				require.Nil(t, err)
				require.Equal(t, c.want, got.Value)
				require.False(t, got.Sensitive)
			}
		})
	}
}
