﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/qconnect/QConnectRequest.h>
#include <aws/qconnect/QConnect_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace QConnect {
namespace Model {

/**
 */
class GetNextMessageRequest : public QConnectRequest {
 public:
  AWS_QCONNECT_API GetNextMessageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetNextMessage"; }

  AWS_QCONNECT_API Aws::String SerializePayload() const override;

  AWS_QCONNECT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The identifier of the Amazon Q in Connect assistant.</p>
   */
  inline const Aws::String& GetAssistantId() const { return m_assistantId; }
  inline bool AssistantIdHasBeenSet() const { return m_assistantIdHasBeenSet; }
  template <typename AssistantIdT = Aws::String>
  void SetAssistantId(AssistantIdT&& value) {
    m_assistantIdHasBeenSet = true;
    m_assistantId = std::forward<AssistantIdT>(value);
  }
  template <typename AssistantIdT = Aws::String>
  GetNextMessageRequest& WithAssistantId(AssistantIdT&& value) {
    SetAssistantId(std::forward<AssistantIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon Q in Connect session.</p>
   */
  inline const Aws::String& GetSessionId() const { return m_sessionId; }
  inline bool SessionIdHasBeenSet() const { return m_sessionIdHasBeenSet; }
  template <typename SessionIdT = Aws::String>
  void SetSessionId(SessionIdT&& value) {
    m_sessionIdHasBeenSet = true;
    m_sessionId = std::forward<SessionIdT>(value);
  }
  template <typename SessionIdT = Aws::String>
  GetNextMessageRequest& WithSessionId(SessionIdT&& value) {
    SetSessionId(std::forward<SessionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next message. Use the value returned in the SendMessage or
   * previous response in the next request to retrieve the next message.</p>
   */
  inline const Aws::String& GetNextMessageToken() const { return m_nextMessageToken; }
  inline bool NextMessageTokenHasBeenSet() const { return m_nextMessageTokenHasBeenSet; }
  template <typename NextMessageTokenT = Aws::String>
  void SetNextMessageToken(NextMessageTokenT&& value) {
    m_nextMessageTokenHasBeenSet = true;
    m_nextMessageToken = std::forward<NextMessageTokenT>(value);
  }
  template <typename NextMessageTokenT = Aws::String>
  GetNextMessageRequest& WithNextMessageToken(NextMessageTokenT&& value) {
    SetNextMessageToken(std::forward<NextMessageTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_assistantId;

  Aws::String m_sessionId;

  Aws::String m_nextMessageToken;
  bool m_assistantIdHasBeenSet = false;
  bool m_sessionIdHasBeenSet = false;
  bool m_nextMessageTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace QConnect
}  // namespace Aws
