﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3-crt/S3CrtRequest.h>
#include <aws/s3-crt/S3Crt_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace S3Crt {
namespace Model {

/**
 */
class ListDirectoryBucketsRequest : public S3CrtRequest {
 public:
  AWS_S3CRT_API ListDirectoryBucketsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListDirectoryBuckets"; }

  AWS_S3CRT_API Aws::String SerializePayload() const override;

  AWS_S3CRT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_S3CRT_API bool HasEmbeddedError(IOStream& body, const Http::HeaderValueCollection& header) const override;
  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_S3CRT_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p> <code>ContinuationToken</code> indicates to Amazon S3 that the list is being
   * continued on buckets in this account with a token.
   * <code>ContinuationToken</code> is obfuscated and is not a real bucket name. You
   * can use this <code>ContinuationToken</code> for the pagination of the list
   * results. </p>
   */
  inline const Aws::String& GetContinuationToken() const { return m_continuationToken; }
  inline bool ContinuationTokenHasBeenSet() const { return m_continuationTokenHasBeenSet; }
  template <typename ContinuationTokenT = Aws::String>
  void SetContinuationToken(ContinuationTokenT&& value) {
    m_continuationTokenHasBeenSet = true;
    m_continuationToken = std::forward<ContinuationTokenT>(value);
  }
  template <typename ContinuationTokenT = Aws::String>
  ListDirectoryBucketsRequest& WithContinuationToken(ContinuationTokenT&& value) {
    SetContinuationToken(std::forward<ContinuationTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Maximum number of buckets to be returned in response. When the number is more
   * than the count of buckets that are owned by an Amazon Web Services account,
   * return all the buckets in response.</p>
   */
  inline int GetMaxDirectoryBuckets() const { return m_maxDirectoryBuckets; }
  inline bool MaxDirectoryBucketsHasBeenSet() const { return m_maxDirectoryBucketsHasBeenSet; }
  inline void SetMaxDirectoryBuckets(int value) {
    m_maxDirectoryBucketsHasBeenSet = true;
    m_maxDirectoryBuckets = value;
  }
  inline ListDirectoryBucketsRequest& WithMaxDirectoryBuckets(int value) {
    SetMaxDirectoryBuckets(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetCustomizedAccessLogTag() const { return m_customizedAccessLogTag; }
  inline bool CustomizedAccessLogTagHasBeenSet() const { return m_customizedAccessLogTagHasBeenSet; }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  void SetCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag = std::forward<CustomizedAccessLogTagT>(value);
  }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  ListDirectoryBucketsRequest& WithCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    SetCustomizedAccessLogTag(std::forward<CustomizedAccessLogTagT>(value));
    return *this;
  }
  template <typename CustomizedAccessLogTagKeyT = Aws::String, typename CustomizedAccessLogTagValueT = Aws::String>
  ListDirectoryBucketsRequest& AddCustomizedAccessLogTag(CustomizedAccessLogTagKeyT&& key, CustomizedAccessLogTagValueT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag.emplace(std::forward<CustomizedAccessLogTagKeyT>(key), std::forward<CustomizedAccessLogTagValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_continuationToken;

  int m_maxDirectoryBuckets{0};

  Aws::Map<Aws::String, Aws::String> m_customizedAccessLogTag;
  bool m_continuationTokenHasBeenSet = false;
  bool m_maxDirectoryBucketsHasBeenSet = false;
  bool m_customizedAccessLogTagHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Crt
}  // namespace Aws
