// SPDX-FileCopyrightText: 2017 Martin Kacej <m.kacej@atlas.sk>
// SPDX-FileCopyrightText: 2023 Devin Lin <devin@kde.org>
// SPDX-License-Identifier: LGPL-2.0-or-later

import QtQuick
import QtQuick.Layouts
import QtQuick.Window
import QtQuick.Controls as Controls

import org.kde.plasma.networkmanagement as PlasmaNM
import org.kde.kirigami as Kirigami
import org.kde.ksvg as KSvg
import org.kde.kirigamiaddons.formcard 1 as FormCard

FormCard.AbstractFormDelegate {
    id: root

    property bool editMode
    property var map : []
    property bool predictableWirelessPassword: !Uuid && Type == PlasmaNM.Enums.Wireless &&
                                                    (SecurityType == PlasmaNM.Enums.StaticWep ||
                                                     SecurityType == PlasmaNM.Enums.WpaPsk ||
                                                     SecurityType == PlasmaNM.Enums.Wpa2Psk ||
                                                     SecurityType == PlasmaNM.Enums.SAE)
    property real rxSpeed: 0
    property real txSpeed: 0

    verticalPadding: Kirigami.Units.largeSpacing

    Timer {
        id: timer
        repeat: true
        interval: 2000
        running: model.ConnectionState === PlasmaNM.Enums.Activated
        triggeredOnStart: true
        // property int can overflow with the amount of bytes.
        property double prevRxBytes: 0
        property double prevTxBytes: 0
        onTriggered: {
            rxSpeed = prevRxBytes === 0 ? 0 : (RxBytes - prevRxBytes) * 1000 / interval
            txSpeed = prevTxBytes === 0 ? 0 : (TxBytes - prevTxBytes) * 1000 / interval
            prevRxBytes = RxBytes
            prevTxBytes = TxBytes
        }
    }

    contentItem: RowLayout {
        spacing: 0

        Item {
            Layout.rightMargin: Kirigami.Units.gridUnit
            implicitWidth: Kirigami.Units.iconSizes.smallMedium
            implicitHeight: Kirigami.Units.iconSizes.smallMedium

            Kirigami.Icon {
                implicitWidth: Kirigami.Units.iconSizes.smallMedium
                implicitHeight: Kirigami.Units.iconSizes.smallMedium
                visible: model.ConnectionState !== PlasmaNM.Enums.Activating
                anchors.centerIn: parent
                source: mobileProxyModel.showSavedMode ? "network-wireless-connected-100" : model.ConnectionIcon
            }

            Controls.BusyIndicator {
                anchors.fill: parent
                running: model.ConnectionState === PlasmaNM.Enums.Activating
            }
        }

        Controls.Label {
            id: internalTextItem
            Layout.fillWidth: true
            text: ItemUniqueName
            elide: Text.ElideRight
            font.bold: model.ConnectionState === PlasmaNM.Enums.Activated
            Accessible.ignored: true // base class sets this text on root already
            textFormat: Text.PlainText
        }

        RowLayout {
            Kirigami.Icon {
                Layout.alignment: Qt.AlignVCenter
                Layout.preferredWidth: Kirigami.Units.iconSizes.smallMedium
                Layout.preferredHeight: Kirigami.Units.iconSizes.smallMedium
                visible: model.ConnectionState === PlasmaNM.Enums.Activated
                source: 'checkmark'
            }

            // ensure that the row is always of same height
            Controls.ToolButton {
                id: heightMetrics
                opacity: 0
                implicitWidth: 0
                icon.name: 'network-connect'
                enabled: false
            }
            Controls.ToolButton {
                icon.name: "network-connect"
                text: i18n("Connect")
                visible: model.ConnectionState != PlasmaNM.Enums.Activated && root.editMode
                display: Controls.ToolButton.IconOnly
                onClicked: changeState()
            }
            Controls.ToolButton {
                icon.name: "network-disconnect"
                text: i18n("Disconnect")
                visible: model.ConnectionState == PlasmaNM.Enums.Activated && root.editMode
                display: Controls.ToolButton.IconOnly
                onClicked: handler.deactivateConnection(model.ConnectionPath, DevicePath)
            }
            Controls.ToolButton {
                icon.name: "configure"
                text: i18n("Configure")
                visible: (Uuid != "") && root.editMode
                display: Controls.ToolButton.IconOnly
                onClicked: {
                    kcm.push("NetworkSettings.qml", {path: model.ConnectionPath})
                }
            }
            Controls.ToolButton {
                icon.name: "entry-delete"
                text: i18n("Delete")
                visible: (Uuid != "") && root.editMode
                display: Controls.ToolButton.IconOnly
                onClicked: handler.removeConnection(model.ConnectionPath)
            }
        }
    }

    onClicked: {
        changeState()
    }

    function changeState() {
        if (Uuid || !predictableWirelessPassword) {
            if (model.ConnectionState == PlasmaNM.Enums.Deactivated) {
                if (!predictableWirelessPassword && !Uuid) {
                    handler.addAndActivateConnection(DevicePath, SpecificPath);
                } else {
                    handler.activateConnection(model.ConnectionPath, DevicePath, SpecificPath);
                }
            } else {
                kcm.push("ConnectionInfo.qml", {details: model.ConnectionDetails,
                                                connectionName: ItemUniqueName,
                                                delegate: root})

            }
        } else if (predictableWirelessPassword) {
            connectionDialog.headingText = i18n("Connect to") + " " + ItemUniqueName;
            connectionDialog.devicePath = DevicePath;
            connectionDialog.specificPath = SpecificPath;
            connectionDialog.securityType = SecurityType;
            connectionDialog.openAndClear();
        }
    }
}
