module JsxListOptimisation.Components


open Fable.Core
open Fable.Core.JsInterop

(*

    This file is used both in the Integration and React tests

    - Integration is used to verify the syntax output
    - React is used to verify that we emit valid JSX which generates the expected HTML

    Also, when running React tests check if there are any warning generated by React like

        Warning: Each child in a list should have a unique "key" prop.

    which indicates a cases where we try to optimise the JSX output.

*)

// Create our own binding for React API, and make a minimal Feliz-like DSL

// Necessary for JSX compilation
let React: obj = importAll "react"

[<ImportMember("react")>]
[<AllowNullLiteral>]
type JSX_ReactElement = interface end

type JSX_IReactProperty = JSX.Prop

[<Erase>]
type prop =
    static member inline children(children: JSX_ReactElement list) : JSX_IReactProperty = "children", children

    static member inline text(value: int) : JSX_IReactProperty = "children", [ JSX.text !!value ]

    static member inline key(value: int) : JSX_IReactProperty = "key", string value

    static member inline className (cls : string) : JSX_IReactProperty = "className", cls

[<Erase>]
type Html =

    static member inline fragment(children: JSX_ReactElement list) : JSX_ReactElement =
        JSX.create "" [ "children" ==> children ] |> unbox

    static member inline div(children: JSX_ReactElement list) : JSX_ReactElement =
        JSX.create "div" [ "children" ==> children ] |> unbox

    static member inline div(text: string) : JSX_ReactElement =
        JSX.create "div" [ "children" ==> JSX.text text ] |> unbox

    static member inline div(props: JSX_IReactProperty list) : JSX_ReactElement = JSX.create "div" props |> unbox

let divWithText =
    Html.div "Test 1"

let divWithNestedList =
    Html.div
        [
            yield! [
                Html.div "Test 1"
                Html.div "Test 2"
            ]
        ]

let divWithMultipleNestedList =
    Html.div [
        yield! [
            Html.div "Test 1"
        ]
        yield! [
            Html.div "Test 2"
        ]
    ]

let divWithMixOfElementAndList =
    Html.div [
        Html.div "Test 1"
        yield! [
            Html.div "Test 2"
        ]
    ]

let multiLevelMixOfElementAndList =
    Html.div [
        Html.div "Test 1"
        yield! [
            Html.div "Test 2"
            Html.div [
                Html.div "Test 2.1"
                yield! [
                    Html.div "Test 2.2"
                ]
            ]
        ]
    ]

// F# compiler can optimise condition sometimes
let divWithOptimisedTrueCondition =
    Html.div [
        if true then
            Html.div "true"
        else
            Html.div "false"
    ]

// F# compiler can optimise condition sometimes
let divWithOptimisedFalseCondition =
    Html.div [
        if false then
            Html.div "true"
        else
            Html.div "false"
    ]

// Prevent compiler optimisation
let divWithConditionalChildren a =
    Html.div [
        if a = "b" then
            Html.div "true"
        else
            Html.div "false"
    ]

let divWithForLoop =
    Html.div [
        Html.div "Hello"
        for i in 0..2 do
            Html.div [
                prop.key i
                prop.text i
            ]
    ]

let divWithFragment =
    Html.div [
        Html.fragment [
            Html.div "Test 1"
        ]
    ]

let divWithAttributes =
    Html.div [
        prop.className "header"
        prop.children [
            Html.div "Test 1"
        ]
    ]

let divWithConditionalWithoutElseBranchWorks show =
    Html.div [
        if show then
            Html.div "I show sometimes!"

        Html.div "I show always!"
    ]

let propsCanUseUnbox =
    Html.div [
        unbox<JSX_IReactProperty> ("id", "myid")
    ]
