#------------------------------------------------------------------
# $Header: /home/projects/cvs/chklogs/bin/Smtp.pm,v 2.1 1997/09/28 20:46:40 grimaldo Exp $
#------------------------------------------------------------------
# AUTHOR: D. Emilio Grimaldo T.
# DATE  : April 1997
# DESCR.
#       This module implements the SMTP protocol for talking to the
#	SMTP port of the mail server. By using this scripts can send
#	properly formatted e-mail messages without having to create a
#	temporary file or knowing the location and command line syntax
#	of the mail transfer agent (Sendmail, Smail, Qmail...).
#	
#	Sending a mail is as simple as:
#		use Smtp;
#		&OpenSMTP(\*SOCKET_HDL, host);
#		&Mail(\*SOCKET_HDL, .... );
#		&CloseConnection(|=\*SOCKET_HDL);
#
#	Additionally this module has a generalized interface for 
#	connecting to any port (i.e. time of day...) such as:
#		use Smtp;
#		&EstablishConnection(\*SOCKET_HDL, host, port, prot_name);
#		 ... i/o with socket ...
#		&CloseConnection(|=\*SOCKET_HDL);
#
# USAGE:
#       use lib '/lib/dir/where/module/resides';
#       use Smtp;
#
# LIMITATIONS:
#	* The SMTP services only retrieve the status/response from the
#	  server but do not actually check for the status code as
#	  defined in the RFC. This is work in progress.
#	* Non-blocking does not work

require 5.003;
package Smtp;
    require Exporter;
    @ISA	= qw(Exporter);
    @EXPORT	= qw(&EstablishConnection &CloseConnection 
                     &OpenSMTP &Mail &SmtpDebug &GethostSMTP &GetVersion);
    use Socket;
    use Fcntl;
    use strict;

#************************************************
#       C O N F I G U R A T I O N
my $Debug = 0;
#************************************************

#
# Local Declarations
#
my $VERSION;
my $ourSMTPserver;

#************************************************
# FUNCTION : BEGIN
# PROTOTYPE: -
# RETURNS  : -
# GLOBALS  : -
# PRIVATES : VERSION 
# DESCRIPTION
#
BEGIN {
    $VERSION = '$Revision: 2.1 $';
    $VERSION =~ m/Revision:\s+(\d+\.\d+\.*\d*\.*\d*)/;
    $VERSION = $1;
}

#************************************************
# FUNCTION : EstablishConnection
# PROTOTYPE: EstablishConnection(\$Socket, $host, $port, $protocol_name)
# RETURNS  : 1 on success, 0 otherwise
# GLOBALS  : -
# PRIVATES : S
# DESCRIPTION
#		Opens a socket connection to the port and
#		connects to it. Switch mode to force flush after
#		every write (non-buffered). 
#		  The protocol_name can be 'tcp' or 'udp'.
sub EstablishConnection {		# PROTO(\$,$,$,$)
    my($sock_id, $host, $port, $proto_name) =  @_;
    my($in_addr, $proto, $addr);
    my $nothing;
    my $rest;
    my $code;

    # Form the server address from hostname and port number
    $in_addr = (gethostbyname($host))[4];
    $addr = sockaddr_in($port, $in_addr);	# Latest Socket version!

    $proto = getprotobyname($proto_name);

    # Create an internet protocol socket
    socket(\$sock_id, AF_INET, SOCK_STREAM, $proto) or return 0;

    # Connect our socket to the server socket
    connect(\$sock_id, $addr) or return 0;

    # Force fflush on socket file handle after every write
    select(\$sock_id); $| = 1; select(STDOUT);
    fcntl \$sock_id, O_NONBLOCK, $nothing;

    recv \$sock_id, $nothing, 100, 0;
    ($code, $ourSMTPserver, $rest) = split(/\s+/, $nothing, 3);
    print "Established: $nothing" if $Debug != 0;
    return 1;
}

#************************************************
# FUNCTION : OpenSMTP
# PROTOTYPE: OpenSMTP( \$Socket, $host )
# RETURNS  : 1 on success, 0 otherwise
# GLOBALS  : -
# PRIVATES : -
# DESCRIPTION
#		Simply does and EstablishConnection to the
#		SMTP port (port 25) using TCP.
#		  Only ONE connection can be active!
sub OpenSMTP {				# PROTO(\$,$)
    my $sock_id = shift;
    my $host = shift;
    my $smtp_port = 25;		# The standard port
    my $result;

    $smtp_port = (getservbyname('smtp','tcp'))[2];
    $result = &EstablishConnection($$sock_id, $host, $smtp_port, 'tcp');
    return $result;
}
    
#************************************************
# FUNCTION : CloseConnection
# PROTOTYPE: CloseConnection(\$Socket)
# RETURNS  : -
# GLOBALS  : -
# PRIVATES : -
# DESCRIPTION
#
sub CloseConnection {			# PROTO(\$)
    my $sock_id = shift;
    close($$sock_id);
}

#************************************************
# FUNCTION : sendSMTP
# PROTOTYPE: sendSMTP( $socket, $msg )
# RETURNS  : -
# GLOBALS  : -
# PRIVATES : -
# DESCRIPTION
#		Sends msg over the socket connection to SMTP
#		port and awaits the server response. This service
#		must *not* be used for the data part otherwise it
#		would block!, so only control messages!
sub sendSMTP {
    my ($socket, $msg) = @_;
    my $result;

    print $socket $msg;
    recv $socket, $result, 100, 0;
    $result =~ s/\r/ /g;
    $result =~ s/\n/ /g;
    print "$msg\t-> $result\n" if $Debug != 0;
}

#************************************************
# FUNCTION : sendSMTPdata
# PROTOTYPE: sendSMTPdata( $socket, $msg )
# RETURNS  : -
# GLOBALS  : -
# PRIVATES : -
# DESCRIPTION
#		Sends msg over the socket connection to the SMTP server.
#		This call is used for the data section of the message
#		because no response is expected from server until we
#		finish.
#		  Replace the \n for \r\n otherwise some SMTP servers
#		would hang waiting for \r\n and QMAIL would refuse to
#		work.
sub sendSMTPdata {
    my ($socket, $msg) = @_;
    my $result;

    $msg =~ s/\n/\r\n/g;	# Otherwise some SMTP servers would hang
    print $socket $msg;
    print "$msg" if $Debug != 0;
}

#************************************************
# FUNCTION : Mail
# PROTOTYPE: Mail(\$Socket, $to, $from, $subject, \@msg, \@extra_header [, \%fake ])
# RETURNS  : -
# GLOBALS  : -
# PRIVATES : 
# DESCRIPTION
#		Utility function to send a mail that was already prepared.
# NOTE:
#	The \r\n seem to be needed instead of just \n, QMail reports that
#	some SMTP servers would hang waiting for \r\n if there is an
#	astray \n.
#	  Aha! don't think you can get with fake.
sub Mail {
    my $Sock_Id = shift;
    my $To      = shift;
    my $From    = shift;
    my $Subject = shift;
    my $msgRef  = shift;	# The actual message content
    my $x_hdr;			# If present, the X-tended headers
    my $hdr_override;		# If present, override host and from fields
    my $lines;
    my $time = localtime;
    my $Home = 'localhost';
    my @tm   = ();

    @tm = split(/\s+/,$time);
    $time = sprintf "%s, %02d %s %s %s", $tm[0], $tm[2], $tm[1],
                                         $tm[4], $tm[3];
    if ($#_ == -1) {		# No extra headers, No overrides
#    	undefine $x_hdr;
#    	undefine $hdr_override;
    }
    elsif ($#_ == 0) {		# Only extra headers
	$x_hdr = shift;
#    	undefine $hdr_override;
    }
    elsif ($#_ == 1) {		# Extra headers and overrides
	$x_hdr = shift;
	$hdr_override = shift;
	$Home = $$hdr_override{'host'} if exists($$hdr_override{'host'});
	$From = $$hdr_override{'from'} if exists($$hdr_override{'from'});
    }

    #
    # Greet the SMTP server
    #
    sendSMTP $$Sock_Id, "HELO $Home\n";

    #
    # Initiate the transaction for this mail message
    # According to RFC we should have either:
    #		From: logname@host.dom.ain (Full name and other comments)
    # or
    #		From: Full Name <logname@host.dom.ain>
    #
    sendSMTP $$Sock_Id, "MAIL FROM:<$From>\n";
    sendSMTP $$Sock_Id, "RCPT TO:<$To>\n";

    sendSMTP $$Sock_Id, "DATA\r\n";
    #
    # Output the necessary mail headers
    #
    sendSMTPdata $$Sock_Id, "From: $From\nTo: $To\nSubject: $Subject\n";
    $lines = $#$msgRef + 1;
    sendSMTPdata $$Sock_Id, "Date: $time\nLines: $lines\n";

    #
    # Put out the extra headers (if any)
    #
    if ((ref($x_hdr) eq 'ARRAY') && $#$x_hdr >= 0) {
	foreach $lines (0 .. $#$x_hdr) {
	    sendSMTPdata $$Sock_Id, $$x_hdr[$lines];
	}
    }
    sendSMTPdata $$Sock_Id, "\n";

    #
    # Put out the body of the mail message
    #
    foreach $lines (0 .. $#$msgRef) {
	sendSMTPdata $$Sock_Id, $$msgRef[$lines];
    }

    sendSMTP $$Sock_Id, ".\r\n";
    sendSMTP $$Sock_Id, "QUIT\n";
}

sub SmtpDebug {
    my $dflag = shift;

    $Debug = $dflag;
}

#************************************************
# FUNCTION : GethostSMTP
# PROTOTYPE: GethostSMTP()
# RETURNS  : fully qualified domain name of SMTP host
# GLOBALS  : -
# PRIVATES : ourSMTPserver
# DESCRIPTION
#
sub GethostSMTP {
    return $ourSMTPserver;
}

sub GetVersion {
    return $VERSION;
}

#************************************************
# FUNCTION : END
# PROTOTYPE: -
# RETURNS  : -
# GLOBALS  : -
# PRIVATES : 
# DESCRIPTION
#
END {
}

1;
