#include <iostream.h>
#include <sadblib.hh>
#include <sadbcmd.hh>
//
// Copyright (C) 1995  Lars Berntzon
//

/******************************************************************
 *		S H O W _ E N T R I E S
 *		-----------------------
 * Description:
 *	This routine handles how entries shall be printed to output.
 *	the algorithm for this is rather complex, at least a lot
 *	of different cases must be handeled. The problem can be
 *	divided into a matrix of conditions:
 *	
 *	N.o columns  |	One machine	     Several entries
 *	to show      |
 *	-------------+ /-------------- to be output ----------------------\
 *	0	     |  machine	              machine
 *	1            |  subval1\nsubval2      machine  subval1;subval2
 *      >1           |  col1=subval1;subval2  machine  col1=subval1;subval2
 *
 * 	subval means that each value can have several subvalues 
 *	separated by semicolons but they shall sometimes not
 *	be shown with semicolons bu on differens rows.
 * 
 * 	Also if the output is to be uniqe, no machine name is to be printed
 *	and only the values with each sub value on a separate row as in the
 *	case with 1 column and one machine above.
 *
 *	To summarize there are 5 cases above which also can be visualized
 *	as follows:
 *
 *	Case#		What to show on output
 *	-----		----------------------
 *	MODE_0		machine
 *	MODE_1		subval1 \n subval2
 *	MODE_2		col1=subval1;subval2 col2=...
 *	MODE_3		machine subval1;subval2
 *	MODE_4		machine col1=subval1;subval2 col2=...
 *
 * 	Criterias for the cases (or modes) are:
 *	Case#		Criteria
 *	-----		--------
 *	MODE_0		No columns to show.
 *	MODE_1		One column to show and one machine, or unique.
 *	MODE_2		Several columns to show and one machine.
 *	MODE_3		One column to show but several entries.
 *	MODE_4		Several columns to show and several entries.
 *
 * Arguments:
 *	entries		- List of entries
 *	showList	- List of columns to show
 *	singleMachine	- Flag, only one machine specified
 *	unique		- Unique mode
 *	outputMode	- Produce normal, shell or semicolon separated syntax.
 *	out		- Output file pointer
 *
 ******************************************************************/

typedef enum {mode_unknown, mode_0, mode_1, mode_2, mode_3, mode_4 } Mode;

void
show_entries(ListList &entries,
             TextList &showList,
             int singleMachine,
             int unique,
             OutputMode outputMode,
             FILE *out)
{
    TextList columns;		// List of columns to show.
    Pix col;			// Index over columns.
    char *str = NULL;		// Temporary char pointer.
    Mode mode = mode_unknown;	// What case above the rutine is using.
    int nColumns = 0;		// Number of columns to show.
    char buf[MAXNAME];		// Temporary string.
    const char *txt;		// Text value to show.
    int firstRow = 1;		// Flag for when first row is printed.
    Pix i;			// Index.

    /*
     * Check how many columns to show, 0, 1 or several (representated by a 2).
     */
    i = showList.first();
    if (i != 0) {
    	nColumns ++;		// At least one column.
	showList.next(i);
    	if (i != 0) {
    	    nColumns++;	// Two or more.
    	}
    }

    /*
     * Now examine what case as described above the routine should
     * use during execution.
     */
    if (nColumns == 0) { 
	mode = mode_0;
    }
    else if ((nColumns == 1 && singleMachine == 1) || unique) {
    	mode = mode_1;
    }
    else if (nColumns == 2 && singleMachine == 1) {
    	mode = mode_2;
    }
    else if (nColumns == 1 && singleMachine == 0) {
    	mode = mode_3;
    }
    else if (nColumns == 2 && singleMachine == 0) {
    	mode = mode_4;
    }

    /*
     * Loop through all entries.
     */
    for(i = entries.first(); i != 0; entries.next(i))
    {
	switch (mode)
	{
	case mode_0:
	    fprintf(out, "%s\n", entries(i).name());
	    break;

	case mode_1:
	    txt = entries(i).list().front().text();
	    if (txt == 0) {
	    	txt = "";
	    }
	    strncpy(buf, txt, sizeof buf);
	    for(str = strtok(buf, ";");
	        str != NULL ;
	        str = strtok(NULL, ";"))
	    {
		fprintf(out, "%s\n", str);
	    }
	    break;

	case mode_3:
	case mode_4:
	case mode_2:
	    columns = entries(i).list();

	    if (outputMode == semi && firstRow) {
		//
		// If semicolon output mode, print the header line first.
		//
		firstRow = 0;

	        fprintf(out, "\"KEY\"");
		for(col = columns.first(); col != 0; columns.next(col))
		{
		    fprintf(out, ";\"%s\"", strupper(columns(col).name()));
		}
		fprintf(out, "\n");
	    }

	    //
	    // In mode 3 and 4 the key shall be printed.
	    //
	    if (mode == mode_3 || mode == mode_4)
	    {
		switch (outputMode)
		{
		case normal:
		case shell:
		    fprintf(out, "%s ", entries(i).name());
		    break;

		case semi:
		    fprintf(out, "\"%s\"", entries(i).name());
		    break;

		case export:
		    fprintf(out, "key %s %s\n", entries(i).name(), entries(i).name());
		    break;
		}
	    }

	    for(col = columns.first(); col != 0; columns.next(col))
	    {
		//
		// Make sure text is not a NULL pointer.
		//
		txt = columns(col).text();
		if (txt == 0) {
		    txt = "";
		}

		switch(outputMode)
		{
		case normal:
		    if (mode == mode_2 || mode == mode_4) {
			//
			// Multiple columns output.
			//
			fprintf(out, "%s=%s ", columns(col).name(), txt);
		    }
		    else {
			//
			// Single column.
			//
			fprintf(out, "%s ", txt);
		    }
		    break;

		case shell:
		    fprintf(out, "%s=%s ", strupper(columns(col).name()), shell_escape(txt));
		    break;
		
		case semi:
		    fprintf(out, ";\"%s\"", quote_escape(txt));
		    break;

		case export:
		    fprintf(out, "%s %s %s\n", columns(col).name(), entries(i).name(), txt);
		    break;
	        }
	    }
	    if (outputMode != export) {
		fprintf(out, "\n");
	    }
	    break;

	default:
	    BUG("not yet implemented");
	}
    }
}
//
// History of changes:
// $Log: show_entries.cc,v $
// Revision 1.10  1997/03/18 20:13:56  lasse
// Created quote_escape.cc
//
// Revision 1.9  1996/09/14 18:33:29  lasse
// Added some things to the TODO and added pargs
//
// Revision 1.8  1995/09/23  13:46:03  lasse
// Imported from remote
//
// Revision 1.1.1.1  1995/09/11  09:22:59  qdtlarb
// THis is version 0.6
//
// Revision 1.7  1995/09/10  20:43:17  lasse
// Added copyright everywhere
//
// Revision 1.6  1995/09/10  19:03:38  lasse
// Corrected removed Log keyword
//
