//
// Copyright (C) 1995  Lars Berntzon
//
extern "C" {
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/MenuButton.h>
}

#include <sadblib.hh>
#include <iostream.h>
#include <Tokenizer.hh>
#include <xsadb.hh>
#include <DBPanel.hh>
#include <DBList.hh>
#include <CreateColumn.hh>

//
// Constructor
//
DBPanel::DBPanel(void) :
    databaseName(0),
    widgetList(0),
    nameList(0),
    top(0),
    widgetFrame(0),
    fields(0),
    nFields(0)
{
}

DBPanel::DBPanel(Widget topshell, const char *name) :
    databaseName(0),
    widgetList(0),
    nameList(0),
    top(0),
    widgetFrame(0),
    fields(0),
    nFields(0)
{
    //
    // Make private copy of database name.
    //
    databaseName = strdup(name);

    //
    // Only create the shell first time.
    //
    top = XtCreatePopupShell(
		    databaseName,		// Name.
		    transientShellWidgetClass,	// Widget class.
		    topshell,			// Parent.
		    NULL,			// Args.
		    0);				// Num args.

    Widget toplevel = XtCreateManagedWidget(
		    "popup",			// Name.
		    boxWidgetClass,		// Widget class.
		    top,			// Parent.
		    NULL,			// Arguments.
		    0);				// Number of arguments.

    create_menus(toplevel);
    
    XtVaCreateManagedWidget(
		    "label",			// Name.
		    labelWidgetClass,		// Widget class.
		    toplevel,			// Parent.
		    XtNlabel,			// Label.
		    databaseName,		// Name of database.
		    0);

    widgetFrame = XtVaCreateManagedWidget(
		      "widgetFrame",			// Name.
		      boxWidgetClass,		// Widget class.
		      toplevel,			// Parent.
		      0);			// Arguments.

    //
    // Update with all widgets for columns
    //
    update();

    //
    // Create buttons.
    //
    Widget buttons = XtCreateManagedWidget(
		    "buttons",			// Name.
		    boxWidgetClass,		// Widget class.
		    toplevel,			// Parent.
		    NULL,			// Arguments.
		    0);				// Number of arguments.
    
    Widget update = XtVaCreateManagedWidget(
			  "update",			// Name.
			  commandWidgetClass,		// Widget class.
			  buttons,			// Parent.
			  NULL);			// Args.
    XtAddCallback(update, XtNcallback, hdl_x11_update, (XtPointer *)this);

    Widget fetch = XtVaCreateManagedWidget(
			  "fetch",			// Name.
			  commandWidgetClass,		// Widget class.
			  buttons,			// Parent.
			  NULL);			// Args.
    XtAddCallback(fetch, XtNcallback, hdl_x11_fetch, (XtPointer *)this);

    Widget list = XtVaCreateManagedWidget(
			  "list",			// Name.
			  commandWidgetClass,		// Widget class.
			  buttons,			// Parent.
			  NULL);			// Args.
    XtAddCallback(list, XtNcallback, hdl_x11_list, (XtPointer *)this);

    Widget remove = XtVaCreateManagedWidget(
			  "remove",			// Name.
			  commandWidgetClass,		// Widget class.
			  buttons,			// Parent.
			  NULL);			// Args.
    XtAddCallback(remove, XtNcallback, hdl_x11_remove, (XtPointer *)this);

    Widget clear = XtVaCreateManagedWidget(
			  "clear",			// Name.
			  commandWidgetClass,		// Widget class.
			  buttons,			// Parent.
			  NULL);			// Args.
    XtAddCallback(clear, XtNcallback, hdl_x11_clear, (XtPointer *)this);

    Widget done = XtVaCreateManagedWidget(
			  "done",			// Name.
			  commandWidgetClass,		// Widget class.
			  buttons,			// Parent.
			  NULL);			// Args.
    XtAddCallback(done, XtNcallback, hdl_x11_done, (XtPointer*)this);

    //
    // Now make this panels listbox and column creation panel.
    //
    listObject = new DBList(topshell, databaseName, this);
    listObject->registerSelectable(this);

    createColumnObject = new CreateColumn(topshell, databaseName, this);
    createColumnObject->registerUpdateable(this);
}

//
// Update the widget when the number of columns has changed.
//
void
DBPanel::update(void)
{
    Widget subFields;
    int count  = 0;
    char widgetName[1024];
    int idx;					// Index into the widget list.
    Pix p;

    int isRealized = XtIsRealized(widgetFrame);
    if (isRealized) XtUnrealizeWidget(widgetFrame);

    //
    // Remove all old widgets and data if the exist.
    //
    for(int i = 0; i < nFields; i++) {
	free(nameList[i]);
    }
    if (nameList) delete [] nameList;
    if (widgetList) delete [] widgetList;
    if (fields) XtDestroyWidget(fields);

    fields = XtVaCreateManagedWidget(
		      "fields",			// Name.
		      boxWidgetClass,		// Widget class.
		      widgetFrame,		// Parent.
		      XtNorientation,
		      XtEhorizontal,
		      0);			// Arguments.

    //
    // Examine which columns there are.
    //
    SADB &database = dbList[databaseName];
    TextList allColumns = database.list_columns();
    TextList columns;

    //
    // Remove all unwanted columns (ending with _date).
    //
    for(p = allColumns.first(); p != 0; allColumns.next(p))
    {
	//
	// Skip the _date columns, they are mostly ment to be internal.
	//
	if (strcmp(allColumns(p).name() + strlen(allColumns(p).name()) - 5, "_date") != 0) {
	    columns.append(allColumns(p));
	}
    }

    //
    // Allocate the widget and the name list. This list shall be one entry
    // larger than the columns list because the key column is not in the list.
    //
    nFields = columns.length() + 1;
    widgetList = new Widget[nFields];
    nameList = new (char *)[nFields];

    //
    // Loop through all columns and create fields. Make sure the first field
    // is the key field though.
    //
    for(idx = 0, p = columns.first(); idx == 0 || p != 0; idx++)
    {
	//
	// Create sub boxes for every 18 fields.
	//
	if (count <= 0) {
	    count = 18;
	    sprintf(widgetName, "subField%d", idx / 18);
	    subFields = XtCreateManagedWidget(
			      widgetName,		// Name.
			      boxWidgetClass,		// Widget class.
			      fields,			// Parent.
			      NULL,			// Arguments.
			      0);			// Number of arguments.

	}
	count--;

	// Create the key field first.
	if (idx == 0) {
	    nameList[idx] = strdup("key");
	}
	else {
	    nameList[idx] = strdup(columns(p).name());
	    // This is a special because the key is not in the columns list.
	    columns.next(p);
	}
	widgetList[idx] = create_one_field(database,
				       subFields,
				       nameList[idx]);
    }

    XtManageChild(widgetFrame);
    if (isRealized) XtRealizeWidget(widgetFrame);
}

//
// Destructor.
//
DBPanel::~DBPanel(void)
{
    if (listObject) delete listObject;
    if (widgetList) delete [] widgetList;

    for(int i = 0; i < nFields; i++) {
	free(nameList[i]);
    }
    delete [] nameList;

    if (top) XtDestroyWidget(top);


    if (databaseName) free(databaseName);
}


void
DBPanel::popup(void)
{
    XtPopup(top, XtGrabNone);
}

void
DBPanel::hdl_done(void)
{
    popdown();
    listObject->popdown();
}

void
DBPanel::popdown(void)
{
    XtPopdown(top);
}

void
DBPanel::select(const char *key)
{
    int i;
    const char *value = 0;
    char buf[1024];

    SADB &database = dbList[databaseName];

    //
    // Loop through all fields.
    //
    strcpy(buf, key);
    for(i = 0; i < nFields; i++)
    {
    	value = database.fetch(buf, nameList[i]);
	if (value == 0) {
	    value = "";
	}

	XtVaSetValues(widgetList[i], "string", value, NULL);
    }
}

void
DBPanel::hdl_list(void)
{
    TextList entries;			// This will be the list of matching entries.
    const char *value;			// Value of widget.
    int i;

    // Refer to the database.
    SADB &database = dbList[databaseName];

    // List all entries in the database.
    entries = database.list_entries();

    // Loop through all fields and select only those that matches value.
    for(i = 0; i <nFields; i++)
    {
	XtVaGetValues(widgetList[i], "string", &value, NULL);

	// If nonzero value, make sure current entry matches.
	if (value && value[0] != 0)
	{
	    TextList retList;
	    database.match(nameList[i], value, entries, retList);
	    entries = retList;
	}
    }

    //
    // Give the list to the listbox.
    //
    listObject->list(entries);
    listObject->popup();
}

void
DBPanel::hdl_update(void)
{
    char key[1024] = "";
    char *key_p;
    char *value;
    int rc;

    SADB &database = dbList[databaseName];

    //
    // The first column must be the key.
    //
    XtVaGetValues(widgetList[0], "string", &key_p, NULL);
    if (key_p == 0 || key_p[0] == 0) {
    	cerr << "No value for key\n";
    	return;
    }
    strcpy(key, key_p);

    //
    // Loop through all other fields.
    //
    for(int i = 0; i < nFields; i++)
    {
	XtVaGetValues(widgetList[i], "string", &value, NULL);

    	rc = database.store(key, nameList[i], value);
	if (rc != 0) {
	    cerr << "failed to store column " << nameList[i] << endl;
	    return;
	}
    }
}

void
DBPanel::hdl_fetch(void)
{
    char *key;

    //
    // The first column must be the key.
    //
    XtVaGetValues(widgetList[0], "string", &key, NULL);
    if (key == 0 || key[0] == 0) {
    	cerr << "No value for key\n";
    	return;
    }

    select(key);
}


//////////////////////////////////////////////////////////////////
//		h d l _ r e m o v e
//		-------------------
// Description:
//	Remove selected entry.
//
//////////////////////////////////////////////////////////////////
void
DBPanel::remove(void)
{
    char key[1024] = "";
    char *key_p;

    SADB &database = dbList[databaseName];

    //
    // The first column must be the key.
    //
    XtVaGetValues(widgetList[0], "string", &key_p, NULL);
    if (key_p == 0 || key_p[0] == 0) {
    	cerr << "No value for key\n";
    	return;
    }
    strcpy(key, key_p);

    //
    // Loop through all other fields.
    //
    for(int i = 0; i < nFields; i++)
    {
    	if (database.remove(key, nameList[i]) != 0) {
    	   cerr << "failed to remove column: " << nameList[i] << endl;
    	   break;
    	}
    }

    //
    // This will just make the picture empty.
    //
    hdl_clear();
}


//////////////////////////////////////////////////////////////////
//		h d l _ r e m o v e
//		-------------------
// Description:
//	Remove multiple entries.
//
//////////////////////////////////////////////////////////////////
void
DBPanel::remove(TextList &entries)
{
    char key[1024] = "";
    Pix n;

    SADB &database = dbList[databaseName];

    //
    // The first column must be the key.
    //
    for(n = entries.first(); n; entries.next(n))
    {
	strcpy(key, entries(n).name());

	//
	// Loop through all other fields.
	//
	for(int i = 0; i < nFields; i++)
	{
	    if (database.remove(key, nameList[i]) != 0) {
		cerr << "failed to remove column: " << nameList[i] << endl;
		break;
	    }
	}
    }

    //
    // This will just make the picture empty.
    //
    hdl_clear();
}

void
DBPanel::hdl_create_column(void)
{
    createColumnObject->popup();
}

void
DBPanel::hdl_clear(void)
{
    //
    // Loop through all fields.
    //
    for(int i = 0; i <nFields; i++)
    {
	XtVaSetValues(widgetList[i], "string", "", NULL);
    }
}

Widget
DBPanel::create_one_field(SADB &database, Widget parent, const char *name)
{
    Dimension len;

    //
    // Get the description if it exist.
    //
    const char *desc = database.fetch("description.text", name);
    if (desc == 0) {
	desc = name;
    }

    //
    // A box around the label and the text.
    //
    Widget box = XtVaCreateManagedWidget(
			  name,			// Name.
			  boxWidgetClass,	// Widget class.
			  parent,		// Parent.
			  XtNorientation,
			  XtEhorizontal,
			  NULL);

    //
    // The label.
    //
    Widget label = XtVaCreateManagedWidget(
			"label",			// Name.
			labelWidgetClass,		// Widget class.
			box,				// Parent.
			XtNlabel,
			desc,
			NULL);

    XtVaGetValues(label, XtNwidth, &len, NULL);
    len = 330 - len;
    if (len <= 100) len = 100;

    // 
    // And the text.
    //
    Widget text = XtVaCreateManagedWidget(
			"text",				// Name.
			asciiTextWidgetClass,		// Widget class.
	  		box,				// Parent.
	  		XtNeditType,
	  		XawtextEdit,
	  		XtNwidth,
	  		len,
	  		NULL);

    return text;
}

void
DBPanel::create_menus(Widget toplevel)
{
    Widget menuBar = XtCreateManagedWidget(
                              "menuBar",                // Name.
                              boxWidgetClass,           // Widget class.
                              toplevel,                 // Parent.
                              NULL,                     // Arguments.
                              0);                       // Number of arguments.

    //
    // The edit menu button
    //
    Widget editMenu = XtVaCreatePopupShell(
    			      "editMenu", 
    			      simpleMenuWidgetClass,
    			      menuBar,
    			      NULL);

    Widget createColumn = XtVaCreateManagedWidget(
    			      "modifyColumn",
    			      smeBSBObjectClass,
    			      editMenu,
    			      NULL);

    XtAddCallback(createColumn, XtNcallback, hdl_x11_create_column, (XtPointer *)this);

    XtVaCreateManagedWidget(
			      "editMenuButton",		// Name.
			      menuButtonWidgetClass,	// Widget class.
			      menuBar,			// Parent.
			      XtNmenuName,		// Arguments.
			      "editMenu",
			      NULL);
}

//
//---------- Beginning of X11 helper routines -----------------
//

void DBPanel::hdl_x11_create_column(Widget w, XtPointer appData, XtPointer callData)
{
    ((DBPanel *)appData)->hdl_create_column();
}

void DBPanel::hdl_x11_done(Widget w, XtPointer appData, XtPointer callData)
{
    ((DBPanel *)appData)->hdl_done();
}

void DBPanel::hdl_x11_update(Widget w, XtPointer appData, XtPointer callData)
{
    ((DBPanel *)appData)->hdl_update();
}

void
DBPanel::hdl_x11_fetch(Widget w, XtPointer appData, XtPointer callData)
{
    ((DBPanel *)appData)->hdl_fetch();
}

void
DBPanel::hdl_x11_list(Widget w, XtPointer appData, XtPointer callData)
{
    ((DBPanel *)appData)->hdl_list();
}

void
DBPanel::hdl_x11_remove(Widget w, XtPointer appData, XtPointer callData)
{
    ((DBPanel *)appData)->remove();
}

void
DBPanel::hdl_x11_clear(Widget w, XtPointer appData, XtPointer callData)
{
    ((DBPanel *)appData)->hdl_clear();
}

//
// History of changes:
// $Log: DBPanel.cc,v $
// Revision 1.7  1997/07/18 20:08:45  lasse
// Removed erroneus complication.
//
// Revision 1.6  1996/09/29 20:45:54  lasse
// Added the remove button to the list panel.
//
// Revision 1.5  1996/09/14 18:33:43  lasse
// Added some things to the TODO and added pargs
//
// Revision 1.4  1996/03/21 18:57:59  lasse
// Backup
//
// Revision 1.3  1996/03/12 19:43:19  lasse
// Checking in from remote.
//
// Revision 1.1  1996/01/22  20:17:22  lasse
// Checking in from mobile
//
//
