﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/securityhub/SecurityHub_EXPORTS.h>
#include <aws/securityhub/model/ActorSession.h>
#include <aws/securityhub/model/ActorUser.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SecurityHub {
namespace Model {

/**
 * <p> Information about the threat actor identified in an Amazon GuardDuty
 * Extended Threat Detection attack sequence. GuardDuty generates an attack
 * sequence finding when multiple events align to a potentially suspicious
 * activity. To receive GuardDuty attack sequence findings in Security Hub, you
 * must have GuardDuty enabled. For more information, see <a
 * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty-extended-threat-detection.html">GuardDuty
 * Extended Threat Detection </a> in the <i>Amazon GuardDuty User
 * Guide</i>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/Actor">AWS
 * API Reference</a></p>
 */
class Actor {
 public:
  AWS_SECURITYHUB_API Actor() = default;
  AWS_SECURITYHUB_API Actor(Aws::Utils::Json::JsonView jsonValue);
  AWS_SECURITYHUB_API Actor& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SECURITYHUB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p> The ID of the threat actor. </p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  Actor& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Contains information about the user credentials used by the threat
   * actor.</p>
   */
  inline const ActorUser& GetUser() const { return m_user; }
  inline bool UserHasBeenSet() const { return m_userHasBeenSet; }
  template <typename UserT = ActorUser>
  void SetUser(UserT&& value) {
    m_userHasBeenSet = true;
    m_user = std::forward<UserT>(value);
  }
  template <typename UserT = ActorUser>
  Actor& WithUser(UserT&& value) {
    SetUser(std::forward<UserT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Contains information about the user session where the activity
   * initiated.</p>
   */
  inline const ActorSession& GetSession() const { return m_session; }
  inline bool SessionHasBeenSet() const { return m_sessionHasBeenSet; }
  template <typename SessionT = ActorSession>
  void SetSession(SessionT&& value) {
    m_sessionHasBeenSet = true;
    m_session = std::forward<SessionT>(value);
  }
  template <typename SessionT = ActorSession>
  Actor& WithSession(SessionT&& value) {
    SetSession(std::forward<SessionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  ActorUser m_user;

  ActorSession m_session;
  bool m_idHasBeenSet = false;
  bool m_userHasBeenSet = false;
  bool m_sessionHasBeenSet = false;
};

}  // namespace Model
}  // namespace SecurityHub
}  // namespace Aws
