﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudformation/CloudFormationRequest.h>
#include <aws/cloudformation/CloudFormation_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudFormation {
namespace Model {

/**
 * <p>The input for <a>ValidateTemplate</a> action.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cloudformation-2010-05-15/ValidateTemplateInput">AWS
 * API Reference</a></p>
 */
class ValidateTemplateRequest : public CloudFormationRequest {
 public:
  AWS_CLOUDFORMATION_API ValidateTemplateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ValidateTemplate"; }

  AWS_CLOUDFORMATION_API Aws::String SerializePayload() const override;

 protected:
  AWS_CLOUDFORMATION_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>Structure that contains the template body with a minimum length of 1 byte and
   * a maximum length of 51,200 bytes.</p> <p>Conditional: You must pass
   * <code>TemplateURL</code> or <code>TemplateBody</code>. If both are passed, only
   * <code>TemplateBody</code> is used.</p>
   */
  inline const Aws::String& GetTemplateBody() const { return m_templateBody; }
  inline bool TemplateBodyHasBeenSet() const { return m_templateBodyHasBeenSet; }
  template <typename TemplateBodyT = Aws::String>
  void SetTemplateBody(TemplateBodyT&& value) {
    m_templateBodyHasBeenSet = true;
    m_templateBody = std::forward<TemplateBodyT>(value);
  }
  template <typename TemplateBodyT = Aws::String>
  ValidateTemplateRequest& WithTemplateBody(TemplateBodyT&& value) {
    SetTemplateBody(std::forward<TemplateBodyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URL of a file that contains the template body. The URL must point to a
   * template (max size: 1 MB) that is located in an Amazon S3 bucket or a Systems
   * Manager document. The location for an Amazon S3 bucket must start with
   * <code>https://</code>.</p> <p>Conditional: You must pass
   * <code>TemplateURL</code> or <code>TemplateBody</code>. If both are passed, only
   * <code>TemplateBody</code> is used.</p>
   */
  inline const Aws::String& GetTemplateURL() const { return m_templateURL; }
  inline bool TemplateURLHasBeenSet() const { return m_templateURLHasBeenSet; }
  template <typename TemplateURLT = Aws::String>
  void SetTemplateURL(TemplateURLT&& value) {
    m_templateURLHasBeenSet = true;
    m_templateURL = std::forward<TemplateURLT>(value);
  }
  template <typename TemplateURLT = Aws::String>
  ValidateTemplateRequest& WithTemplateURL(TemplateURLT&& value) {
    SetTemplateURL(std::forward<TemplateURLT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_templateBody;

  Aws::String m_templateURL;
  bool m_templateBodyHasBeenSet = false;
  bool m_templateURLHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFormation
}  // namespace Aws
