---
stage: Verify
group: Pipeline Authoring
info: To determine the technical writer assigned to the Stage/Group associated with this page, see https://handbook.gitlab.com/handbook/product/ux/technical-writing/#assignments
title: Caching in GitLab CI/CD
---

{{< details >}}

- Tier: Free, Premium, Ultimate
- Offering: GitLab.com, GitLab Self-Managed, GitLab Dedicated

{{< /details >}}

A cache is one or more files a job downloads and saves. Subsequent jobs that use
the same cache don't have to download the files again, so they execute more quickly.

To learn how to define the cache in your `.gitlab-ci.yml` file,
see the [`cache` reference](../yaml/_index.md#cache).

For advanced cache key strategies, you can use:

- [`cache:key:files`](../yaml/_index.md#cachekeyfiles): Generate keys linked to the content of specific files.
- [`cache:key:files_commits`](../yaml/_index.md#cachekeyfiles_commits): Generate keys linked to the latest commit of specific files.

For more use cases and examples, see [CI/CD caching examples](examples.md).

## How cache is different from artifacts

Use cache for dependencies, like packages you download from the internet.
Cache is stored where GitLab Runner is installed and uploaded to S3 if
[distributed cache is enabled](https://docs.gitlab.com/runner/configuration/autoscale.html#distributed-runners-caching).

Use artifacts to pass intermediate build results between stages.
Artifacts are generated by a job, stored in GitLab, and can be downloaded.

Both artifacts and caches define their paths relative to the project directory, and
can't link to files outside it.

### Cache

- Define cache per job by using the `cache` keyword. Otherwise it is disabled.
- Subsequent pipelines can use the cache.
- Subsequent jobs in the same pipeline can use the cache, if the dependencies are identical.
- Different projects cannot share the cache.
- By default, protected and non-protected branches [do not share the cache](#cache-key-names). However, you can [change this behavior](#use-the-same-cache-for-all-branches).

### Artifacts

- Define artifacts per job.
- Subsequent jobs in later stages of the same pipeline can use artifacts.
- Artifacts expire after 30 days by default. You can define a custom [expiration time](../yaml/_index.md#artifactsexpire_in).
- The latest artifacts do not expire if [keep latest artifacts](../jobs/job_artifacts.md#keep-artifacts-from-most-recent-successful-jobs) is enabled.
- Use [dependencies](../yaml/_index.md#dependencies) to control which jobs fetch the artifacts.

## Good caching practices

To ensure maximum availability of the cache, do one or more of the following:

- [Tag your runners](../runners/configure_runners.md#control-jobs-that-a-runner-can-run) and use the tag on jobs
  that share the cache.
- [Use runners that are only available to a particular project](../runners/runners_scope.md#prevent-a-project-runner-from-being-enabled-for-other-projects).
- [Use a `key`](../yaml/_index.md#cachekey) that fits your workflow. For example,
  you can configure a different cache for each branch.

For runners to work with caches efficiently, you must do one of the following:

- Use a single runner for all your jobs.
- Use multiple runners that have
  [distributed caching](https://docs.gitlab.com/runner/configuration/autoscale.html#distributed-runners-caching),
  where the cache is stored in S3 buckets. Instance runners on GitLab.com behave this way. These runners can be in autoscale mode,
  but they don't have to be. To manage cache objects,
  apply lifecycle rules to delete the cache objects after a period of time.
  Lifecycle rules are available on the object storage server.
- Use multiple runners with the same architecture and have these runners
  share a common network-mounted directory to store the cache. This directory should use NFS or something similar.
  These runners must be in autoscale mode.

## Use multiple caches

You can have a maximum of four caches, per job:

```yaml
test-job:
  stage: build
  cache:
    - key:
        files:
          - Gemfile.lock
      paths:
        - vendor/ruby
    - key:
        files:
          - yarn.lock
      paths:
        - .yarn-cache/
  script:
    - bundle config set --local path 'vendor/ruby'
    - bundle install
    - yarn install --cache-folder .yarn-cache
    - echo Run tests...
```

If multiple caches are combined with a fallback cache key,
the global fallback cache is fetched every time a cache is not found.

## Use a fallback cache key

### Per-cache fallback keys

{{< history >}}

- [Introduced](https://gitlab.com/gitlab-org/gitlab/-/merge_requests/110467) in GitLab 16.0

{{< /history >}}

Each cache entry supports up to five fallback keys with the [`fallback_keys` keyword](../yaml/_index.md#cachefallback_keys).
When a job does not find a cache key, the job attempts to retrieve a fallback cache instead.
Fallback keys are searched in order until a cache is found. If no cache is found,
the job runs without using a cache. For example:

```yaml
test-job:
  stage: build
  cache:
    - key: cache-$CI_COMMIT_REF_SLUG
      fallback_keys:
        - cache-$CI_DEFAULT_BRANCH
        - cache-default
      paths:
        - vendor/ruby
  script:
    - bundle config set --local path 'vendor/ruby'
    - bundle install
    - echo Run tests...
```

In this example:

1. The job looks for the `cache-$CI_COMMIT_REF_SLUG` cache.
1. If `cache-$CI_COMMIT_REF_SLUG` is not found, the job looks for `cache-$CI_DEFAULT_BRANCH`
   as a fallback option.
1. If `cache-$CI_DEFAULT_BRANCH` is also not found, the job looks for `cache-default`
   as a second fallback option.
1. If none are found, the job downloads all the Ruby dependencies without using a cache,
   but creates a new cache for `cache-$CI_COMMIT_REF_SLUG` when the job completes.

Fallback keys follow the same processing logic as `cache:key`:

- If you [clear caches manually](#clear-the-cache-manually), per-cache fallback keys are appended
  with an index like other cache keys.
- If the [**Use separate caches for protected branches** setting](#cache-key-names) is enabled,
  per-cache fallback keys are appended with `-protected` or `-non_protected`.

### Global fallback key

You can use the `$CI_COMMIT_REF_SLUG` [predefined variable](../variables/predefined_variables.md)
to specify your [`cache:key`](../yaml/_index.md#cachekey). For example, if your
`$CI_COMMIT_REF_SLUG` is `test`, you can set a job to download cache that's tagged with `test`.

If a cache with this tag is not found, you can use `CACHE_FALLBACK_KEY` to
specify a cache to use when none exists.

In the following example, if the `$CI_COMMIT_REF_SLUG` is not found, the job uses the key defined
by the `CACHE_FALLBACK_KEY` variable:

```yaml
variables:
  CACHE_FALLBACK_KEY: fallback-key

job1:
  script:
    - echo
  cache:
    key: "$CI_COMMIT_REF_SLUG"
    paths:
      - binaries/
```

The order of caches extraction is:

1. Retrieval attempt for `cache:key`
1. Retrieval attempts for each entry in order in `fallback_keys`
1. Retrieval attempt for the global fallback key in `CACHE_FALLBACK_KEY`

The cache extraction process stops after the first successful cache is retrieved.

## Disable cache for specific jobs

If you define the cache globally, each job uses the
same definition. You can override this behavior for each job.

To disable it completely for a job, use an empty list:

```yaml
job:
  cache: []
```

## Inherit global configuration, but override specific settings per job

You can override cache settings without overwriting the global cache by using
[anchors](../yaml/yaml_optimization.md#anchors). For example, if you want to override the
`policy` for one job:

```yaml
default:
  cache: &global_cache
    key: $CI_COMMIT_REF_SLUG
    paths:
      - node_modules/
      - public/
      - vendor/
    policy: pull-push

job:
  cache:
    # inherit all global cache settings
    <<: *global_cache
    # override the policy
    policy: pull
```

For more information, see [`cache: policy`](../yaml/_index.md#cachepolicy).

## Cache key names

{{< history >}}

- [Introduced](https://gitlab.com/gitlab-org/gitlab/-/issues/330047) in GitLab 15.0.
- `-protected` suffix for Maintainer role and higher [introduced](https://about.gitlab.com/releases/2025/11/26/patch-release-gitlab-18-6-1-released/) in GitLab 18.4.5.

{{< /history >}}

A suffix is added to the cache key, with the exception of the [global fallback cache key](#global-fallback-key).

Cache keys receive the `-protected` suffix if the pipeline:

- Runs for a protected branch or tag. The user must have permission to merge to the
  [protected branch](../../user/project/repository/branches/protected.md) or permission to
  create a [protected tag](../../user/project/protected_tags.md).
- Was started by a user with at least the Maintainer role.

Keys generated in other pipelines receive the `non_protected` suffix.

For example, if:

- `cache:key` is set to `$CI_COMMIT_REF_SLUG`.
- `main` is a protected branch.
- `feature` is an unprotected branch.

| Branch      | Developer role cache key | Maintainer role cache key |
|-------------|--------------------------|---------------------------|
| `main`      | `main-protected`         | `main-protected`          |
| `feature`   | `feature-non_protected`  | `feature-protected`       |

Additionally, for pipelines for tags, the tag's protection status takes precedence for the suffix,
not the branch where the pipeline executes. This behavior ensures consistent security boundaries,
because the triggering reference determines cache access permissions.

For example, if:

- `cache:key` is set to `$CI_COMMIT_TAG`.
- `main` is a protected branch.
- `feature` is an unprotected branch.
- `1.0.0` is a protected tag.
- `1.1.1-rc1` is an unprotected tag.

| Tag         | Branch    | Developer role cache key  | Maintainer role cache key |
|-------------|-----------|---------------------------|---------------------------|
| `1.0.0`     | `main`    | `1.0.0-protected`         | `1.0.0-protected`         |
| `1.0.0`     | `feature` | `1.0.0-protected`         | `1.0.0-protected`         |
| `1.1.1-rc1` | `main`    | `1.1.1-rc1-non_protected` | `1.1.1-rc1-protected`     |
| `1.1.1-rc1` | `feature` | `1.1.1-rc1-non_protected` | `1.1.1-rc1-protected`     |

### Use the same cache for all branches

{{< history >}}

- [Introduced](https://gitlab.com/gitlab-org/gitlab/-/issues/361643) in GitLab 15.0.

{{< /history >}}

If you do not want to use [cache key names](#cache-key-names),
you can have all branches (protected and unprotected) use the same cache.

The cache separation with [cache key names](#cache-key-names) is a security feature
and should only be disabled in an environment where all users with Developer role are highly trusted.

To use the same cache for all branches:

1. On the top bar, select **Search or go to** and find your project.
1. Select **Settings** > **CI/CD**.
1. Expand **General pipelines**.
1. Clear the **Use separate caches for protected branches** checkbox.
1. Select **Save changes**.

## Availability of the cache

Caching is an optimization, but it isn't guaranteed to always work. You might need
to regenerate cached files in each job that needs them.

After you define a [cache in `.gitlab-ci.yml`](../yaml/_index.md#cache),
the availability of the cache depends on:

- The runner's executor type.
- Whether different runners are used to pass the cache between jobs.

### Where the caches are stored

All caches defined for a job are archived in a single `cache.zip` file.
The runner configuration defines where the file is stored. By default, the cache
is stored on the machine where GitLab Runner is installed. The location also depends on the type of executor.

| Runner executor        | Default path of the cache |
| ---------------------- | ------------------------- |
| [Shell](https://docs.gitlab.com/runner/executors/shell.html) | Locally, under the `gitlab-runner` user's home directory: `/home/gitlab-runner/cache/<user>/<project>/<cache-key>/cache.zip`. |
| [Docker](https://docs.gitlab.com/runner/executors/docker.html) | Locally, under [Docker volumes](https://docs.gitlab.com/runner/executors/docker.html#configure-directories-for-the-container-build-and-cache): `/var/lib/docker/volumes/<volume-id>/_data/<user>/<project>/<cache-key>/cache.zip`. |
| [Docker Machine](https://docs.gitlab.com/runner/executors/docker_machine.html) (autoscale runners) | The same as the Docker executor. |

If you use cache and artifacts to store the same path in your jobs, the cache might
be overwritten because caches are restored before artifacts.

### How archiving and extracting works

This example shows two jobs in two consecutive stages:

```yaml
stages:
  - build
  - test

default:
  cache:
    key: build-cache
    paths:
      - vendor/
  before_script:
    - echo "Hello"

job A:
  stage: build
  script:
    - mkdir vendor/
    - echo "build" > vendor/hello.txt
  after_script:
    - echo "World"

job B:
  stage: test
  script:
    - cat vendor/hello.txt
```

If one machine has one runner installed, then all jobs for your project
run on the same host:

1. Pipeline starts.
1. `job A` runs.
1. The cache is extracted (if found).
1. `before_script` is executed.
1. `script` is executed.
1. `after_script` is executed.
1. `cache` runs and the `vendor/` directory is zipped into `cache.zip`.
   This file is then saved in the directory based on the
   [runner's setting](#where-the-caches-are-stored) and the `cache: key`.
1. `job B` runs.
1. The cache is extracted (if found).
1. `before_script` is executed.
1. `script` is executed.
1. Pipeline finishes.

By using a single runner on a single machine, you don't have the issue where
`job B` might execute on a runner different from `job A`. This setup guarantees the
cache can be reused between stages. It only works if the execution goes from the `build` stage
to the `test` stage in the same runner/machine. Otherwise, the cache [might not be available](#cache-mismatch).

During the caching process, there's also a couple of things to consider:

- If some other job, with another cache configuration had saved its
  cache in the same zip file, it is overwritten. If the S3 based shared cache is
  used, the file is additionally uploaded to S3 to an object based on the cache
  key. So, two jobs with different paths, but the same cache key, overwrites
  their cache.
- When extracting the cache from `cache.zip`, everything in the zip file is
  extracted in the job's working directory (usually the repository which is
  pulled down), and the runner doesn't mind if the archive of `job A` overwrites
  things in the archive of `job B`.

It works this way because the cache created for one runner
often isn't valid when used by a different one. A different runner may run on a
different architecture (for example, when the cache includes binary files). Also,
because the different steps might be executed by runners running on different
machines, it is a safe default.

## Clearing the cache

Runners use [cache](../yaml/_index.md#cache) to speed up the execution
of your jobs by reusing existing data. This can sometimes lead to
inconsistent behavior.

There are two ways to start with a fresh copy of the cache.

### Clear the cache by changing `cache:key`

Change the value for `cache: key` in your `.gitlab-ci.yml` file.
The next time the pipeline runs, the cache is stored in a different location.

### Clear the cache manually

You can clear the cache in the GitLab UI:

1. On the top bar, select **Search or go to** and find your project.
1. Select **Build** > **Pipelines**.
1. In the upper-right corner, select **Clear runner caches**.

On the next commit, your CI/CD jobs use a new cache.

> [!note]
> Each time you clear the cache manually, the [internal cache name](#where-the-caches-are-stored) is updated. The name uses the format `cache-<index>`, and the index increments by one. The old cache is not deleted. You can manually delete these files from the runner storage.

## Troubleshooting

### Cache mismatch

If you have a cache mismatch, follow these steps to troubleshoot.

| Reason for a cache mismatch | How to fix it |
| --------------------------- | ------------- |
| You use multiple standalone runners (not in autoscale mode) attached to one project without a shared cache. | Use only one runner for your project or use multiple runners with distributed cache enabled. |
| You use runners in autoscale mode without a distributed cache enabled. | Configure the autoscale runner to use a distributed cache. |
| The machine the runner is installed on is low on disk space or, if you've set up distributed cache, the S3 bucket where the cache is stored doesn't have enough space. | Make sure you clear some space to allow new caches to be stored. There's no automatic way to do this. |
| You use the same `key` for jobs where they cache different paths. | Use different cache keys so that the cache archive is stored to a different location and doesn't overwrite wrong caches. |
| You have not enabled the [distributed runner caching on your runners](https://docs.gitlab.com/runner/configuration/autoscale.html#distributed-runners-caching). | Set `Shared = false` and re-provision your runners. |

#### Cache mismatch example 1

If you have only one runner assigned to your project, the cache
is stored on the runner's machine by default.

If two jobs have the same cache key but a different path, the caches can be overwritten.
For example:

```yaml
stages:
  - build
  - test

job A:
  stage: build
  script: make build
  cache:
    key: same-key
    paths:
      - public/

job B:
  stage: test
  script: make test
  cache:
    key: same-key
    paths:
      - vendor/
```

1. `job A` runs.
1. `public/` is cached as `cache.zip`.
1. `job B` runs.
1. The previous cache, if any, is unzipped.
1. `vendor/` is cached as `cache.zip` and overwrites the previous one.
1. The next time `job A` runs it uses the cache of `job B` which is different
   and thus isn't effective.

To fix this issue, use different `keys` for each job.

#### Cache mismatch example 2

In this example, you have more than one runner assigned to your
project, and distributed cache is not enabled.

The second time the pipeline runs, you want `job A` and `job B` to re-use their cache (which in this case
is different):

```yaml
stages:
  - build
  - test

job A:
  stage: build
  script: build
  cache:
    key: keyA
    paths:
      - vendor/

job B:
  stage: test
  script: test
  cache:
    key: keyB
    paths:
      - vendor/
```

Even if the `key` is different, the cached files might get "cleaned" before each
stage if the jobs run on different runners in subsequent pipelines.

### Concurrent runners missing local cache

If you have configured multiple concurrent runners with the Docker executor, locally cached files might
not be present for concurrently-running jobs as you expect. The names of cache volumes are constructed
uniquely for each runner instance, so files cached by one runner instance are not found in the cache by another runner
instance.

To share the cache between concurrent runners, you can either:

- Use the `[runners.docker]` section of the runners' `config.toml` to configure a single mount point on the host that
  is mapped to `/cache` in each container, such as `volumes = ["/mnt/gitlab-runner/cache-for-all-concurrent-jobs:/cache"]`.
  This approach prevents the runner from creating unique volume names for concurrent jobs.
- Use a distributed cache.
