# frozen_string_literal: true

require 'digest'
require 'openssl'

module GitlabQuality
  module TestTooling
    module Report
      module GroupIssues
        class ErrorMessageNormalizer
          NORMALIZATION_PATTERNS = [
            { pattern: /\d{4}-\d{2}-\d{2}T?[ ]?\d{2}:\d{2}:\d{2}(\.\d+)?Z?/, replacement: "<TIMESTAMP>" },
            { pattern: /[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}/i, replacement: "<UUID>" },
            { pattern: /Correlation Id: [\w]+/, replacement: "Correlation Id: <UUID>" },
            { pattern: /Fabrication of QA::Resource::[A-Za-z:]+/, replacement: "Fabrication of QA::Resource::<RESOURCE>" },
            { pattern: /\b\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}(:\d+)?\b/, replacement: "<IP>" },
            { pattern: /user\d+/, replacement: "<USER>" },
            { pattern: /group\d+/, replacement: "<GROUP>" },
            { pattern: /project\d+/, replacement: "<PROJECT>" },
            { pattern: %r{https?://[^/\s]+/[^\s]*}, replacement: "<URL>" },
            { pattern: %r{/tmp/[^\s]+}, replacement: "<TMPFILE>" },
            { pattern: %r{/var/[^\s]+}, replacement: "<VARFILE>" },
            { pattern: /token=[^\s&]+/, replacement: "token=<TOKEN>" },
            { pattern: /after \d+ seconds/, replacement: "after <N> seconds" },
            { pattern: /waited \d+ seconds/, replacement: "waited <N> seconds" },
            { pattern: /\d+ attempts?/, replacement: "<N> attempts" },
            { pattern: /\s+/, replacement: " " }
          ].freeze

          def normalize(message)
            return "" if message.nil? || message.empty?

            result = message.dup.strip

            NORMALIZATION_PATTERNS.each do |pattern_rule|
              result.gsub!(pattern_rule[:pattern], pattern_rule[:replacement])
            end

            result.strip
          end

          def create_fingerprint(normalized_message)
            OpenSSL::Digest::SHA256.hexdigest(normalized_message.downcase)[0..15]
          end
        end
      end
    end
  end
end
