# frozen_string_literal: true

require 'time'

module GitlabQuality
  module TestTooling
    module Report
      module GroupIssues
        class IssueFinder
          DEFAULT_MAX_AGE_HOURS = 24
          ISSUES_PER_PAGE = 50

          def initialize(client, options = {})
            @client = client
            @options = options
            @token = options[:token]
            @project_id = options[:target_project] || ENV['RESULTS_ISSUE_PROJECT'] || ENV.fetch('CI_PROJECT_ID', nil)
          end

          def find_existing_issue(grouped_failure)
            find_related_issue_by_fingerprint(
              fingerprint: grouped_failure[:fingerprint],
              max_age_hours: DEFAULT_MAX_AGE_HOURS
            )
          end

          private

          def find_related_issue_by_fingerprint(fingerprint:, max_age_hours:)
            Runtime::Logger.info "Searching for existing issue with fingerprint: #{fingerprint}"

            begin
              issues = fetch_recent_open_issues(max_age_hours)
              matching_issue = find_issue_with_fingerprint(issues, fingerprint)
              if matching_issue
                Runtime::Logger.info "Found existing issue: ##{matching_issue.iid} - #{matching_issue.title}"
                convert_to_struct(matching_issue)
              end
            rescue Gitlab::Error::Error => e
              Runtime::Logger.error "GitLab API error searching for issues: #{e.message}"
              nil
            rescue StandardError => e
              Runtime::Logger.error "Error searching for existing issues: #{e.message}"
              nil
            end
          end

          def fetch_recent_open_issues(max_age_hours)
            cutoff_time = Time.now - (max_age_hours * 3600)

            @client.find_issues(options: {
              state: 'opened',
              created_after: cutoff_time.utc.iso8601,
              per_page: ISSUES_PER_PAGE,
              order_by: 'created_at',
              sort: 'desc'
            })
          end

          def find_issue_with_fingerprint(issues, fingerprint)
            fingerprint_tag = "grouped-failure-fingerprint:#{fingerprint}"
            issues.find { |issue| issue.description&.include?(fingerprint_tag) }
          end

          def convert_to_struct(obj)
            return obj unless obj.is_a?(Hash)

            struct_class = Struct.new(*obj.keys.map(&:to_sym))
            struct = struct_class.new

            obj.each do |key, value|
              struct[key.to_sym] = value.is_a?(Hash) ? convert_to_struct(value) : value
            end

            struct
          end
        end
      end
    end
  end
end
