% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data_io.R
\name{read_weights_matrix}
\alias{read_weights_matrix}
\title{Read a weights matrix from an Excel file}
\usage{
read_weights_matrix(path_weights)
}
\arguments{
\item{path_weights}{Character path to the weights Excel file.}
}
\value{
A \code{list} with:
\describe{
  \item{\code{P}}{\eqn{T \times K} numeric matrix of prior weights (rows sum to 1).}
  \item{\code{industries}}{Character vector of sector names (length \eqn{K}).}
  \item{\code{years}}{Integer vector of years (length \eqn{T}).}
}
}
\description{
Loads a sector-by-year weight table, normalizes weights to the simplex per year,
and returns a list with the \eqn{T \times K} prior matrix \code{P}, the sector
names, and the year vector. The first column is assumed to contain sector names
(renamed to \code{Industry}); all other columns are treated as years.
}
\details{
\strong{Expected layout.} One sheet with:
\itemize{
  \item First column: sector names (any header; renamed to \code{Industry}).
  \item Remaining columns: years; the function extracts a 4-digit year from each
        header using \code{stringr::str_extract(Year, "\\\\d{4}")}.
}
Values are parsed with \code{to_num_commas()}, missing rows are dropped, and
weights are normalized within each year to sum to 1. Any absent (sector, year)
entry becomes 0 when pivoting wide. Finally, rows are re-normalized with
\code{row_norm1()} for numerical safety.

\strong{Safeguards.}
\itemize{
  \item Rows with all-missing/zero after parsing are dropped by the filters.
  \item If no valid year columns are found, the function errors.
}
}
\examples{
\donttest{
# Create a temporary Excel file with sample weights
temp_file <- tempfile(fileext = ".xlsx")
df_sample <- data.frame(
  Sector = c("Agriculture", "Manufacturing", "Services", "Construction"),
  "2019" = c(0.20, 0.35, 0.30, 0.15),
  "2020" = c(0.18, 0.37, 0.32, 0.13),
  "2021" = c(0.17, 0.38, 0.33, 0.12),
  "2022" = c(0.16, 0.39, 0.34, 0.11),
  check.names = FALSE
)
openxlsx::write.xlsx(df_sample, temp_file)

# Read the weights matrix
w <- read_weights_matrix(temp_file)

# Inspect structure
str(w)
print(w$P)

# Verify properties
stopifnot(
  is.matrix(w$P),
  nrow(w$P) == 4,  # 4 years
  ncol(w$P) == 4,  # 4 sectors
  all(abs(rowSums(w$P) - 1) < 1e-10),  # rows sum to 1
  length(w$industries) == 4,
  length(w$years) == 4
)

# Clean up
unlink(temp_file)
}

}
\seealso{
\code{\link{read_cpi}}, \code{\link{bayesian_disaggregate}}
}
