% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_fit.R
\name{rsdc_hamilton}
\alias{rsdc_hamilton}
\title{Hamilton Filter (Fixed P or TVTP)}
\usage{
rsdc_hamilton(y, X = NULL, beta = NULL, rho_matrix, K, N, P = NULL)
}
\arguments{
\item{y}{Numeric matrix \eqn{T \times K} of observations (e.g., standardized residuals/returns).
Columns are treated as mean-zero with unit variance; only the correlation structure is modeled.}

\item{X}{Optional numeric matrix \eqn{T \times p} of covariates for TVTP. Required if \code{beta} is supplied.}

\item{beta}{Optional numeric matrix \eqn{N \times p}. TVTP coefficients; row \eqn{i} governs
persistence of regime \eqn{i} via \code{plogis(X[t, ] \%*\% beta[i, ])}.}

\item{rho_matrix}{Numeric matrix \eqn{N \times C} of regime correlation parameters, where
\eqn{C = K(K-1)/2}. Each row is the lower-triangular part (by \code{lower.tri}) of a regime's
correlation matrix.}

\item{K}{Integer. Number of observed series (columns of \code{y}).}

\item{N}{Integer. Number of regimes.}

\item{P}{Optional \eqn{N \times N} fixed transition matrix. Used only when \code{X} or \code{beta} is \code{NULL}.}
}
\value{
A list with:
\describe{
\item{filtered_probs}{\eqn{N \times T} matrix of filtered probabilities
\eqn{\Pr(S_t = j \mid \Omega_t)}.}
\item{smoothed_probs}{\eqn{N \times T} matrix of smoothed probabilities
\eqn{\Pr(S_t = j \mid \Omega_T)}.}
\item{log_likelihood}{Scalar log-likelihood of the model given \code{y}.}
}
}
\description{
Runs the Hamilton (1989) filter for a multivariate regime-switching \emph{correlation} model.
Supports either a fixed (time-invariant) transition matrix \eqn{P} or time-varying transition
probabilities (TVTP) built from exogenous covariates \code{X} via a logistic link.
Returns filtered/smoothed regime probabilities and the log-likelihood.
}
\details{
\itemize{
\item \strong{Correlation rebuild:} For regime \eqn{m}, a correlation matrix \eqn{R_m} is reconstructed
from \code{rho_matrix[m, ]} (lower-triangular fill + symmetrization). Non-PD proposals are penalized.
\item \strong{Transition dynamics:}
\itemize{
\item \emph{Fixed P:} If \code{X} or \code{beta} is missing, a constant \eqn{P} is used
(user-provided via \code{P}; otherwise uniform \eqn{1/N} rows).
\item \emph{TVTP:} With \code{X} and \code{beta}, diagonal entries use
\code{plogis(X[t, ] \%*\% beta[i, ])}. Off-diagonals are equal and sum to \eqn{1 - p_{ii,t}}.
For \eqn{N=1}, \eqn{P_t = [1]}.
}
\item \strong{Numerical safeguards:} A small ridge is added before inversion; if filtering
degenerates at a time step, \code{log_likelihood = -Inf} is returned.
}
}
\note{
TVTP uses a logistic link on the diagonal; off-diagonals are equal by construction.
}
\examples{
set.seed(1)
T <- 50; K <- 3; N <- 2
y <- scale(matrix(rnorm(T * K), T, K), center = TRUE, scale = TRUE)

# Example rho: two regimes with different average correlations
rho <- rbind(c(0.10, 0.05, 0.00),
             c(0.60, 0.40, 0.30))  # lower-tri order for K=3

# Fixed-P filtering
Pfix <- matrix(c(0.9, 0.1,
                 0.2, 0.8), nrow = 2, byrow = TRUE)
out_fix <- rsdc_hamilton(y = y, X = NULL, beta = NULL,
                         rho_matrix = rho, K = K, N = N, P = Pfix)
str(out_fix$filtered_probs)

# TVTP filtering (include an intercept yourself)
X <- cbind(1, scale(seq_len(T)))
beta <- rbind(c(1.2, 0.0),
              c(0.8, -0.1))
out_tvtp <- rsdc_hamilton(y = y, X = X, beta = beta,
                          rho_matrix = rho, K = K, N = N)
out_tvtp$log_likelihood

}
\references{
\insertRef{hamilton1989}{RSDC}
}
\seealso{
\code{\link{rsdc_likelihood}} and \code{\link{rsdc_estimate}}.
}
