#' Davies-Bouldin index
#'
#'
#' This function calculates the Davies-Bouldin index as is defined by Davies and Bouldin (1979) without imposing that the use of the euclidean distance. This function allows calculating the Davies-Bouldin index using different distances.
#'
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param FHW_output List. List with:
#'
#' \itemize{
#'   \item centers: the information of the centers updated.
#'   \item grouping: the information of the groups updated. List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#'    }
#'
#' @return Returns a number, the value of the Davies-Bouldin index.
#'
#'
#'
#' @examples
#'
#'set.seed(451)
#'data=rbind(matrix(runif(20,1,5), nrow = 2, ncol = 10),
#'           matrix(runif(20,20,30), nrow = 2, ncol = 10),
#'           matrix(runif(20,50,70), nrow = 2, ncol = 10))
#'k=3
#'seed=5
#'
#'
#'FHW_output=Hartigan_and_Wong(data,
#'                             Euclideandistance,
#'                             k,
#'                             centers_function_mean,
#'                             init_centers_random,
#'                             seed=seed,
#'                             10)
#'
#'DaviesBouldinIndex(data, FHW_output, Euclideandistance)
#'
#'@references  Davies, D. L., & Bouldin, D. W. (1979). A cluster separation measure. IEEE transactions on pattern analysis and machine intelligence, (2), 224-227.
#'
#' @export



DaviesBouldinIndex<-function(data, FHW_output, distance){

  #S_{i}
  EC=rep(0, length(FHW_output$grouping))
  for (j in 1:length(FHW_output$grouping)){

    PointsGj=FHW_output$grouping[[j]]
    centerGj=FHW_output$centers[j,]

    dm=rep(0,length(PointsGj))
    for (m in 1:length(PointsGj)){
      dm[m]=distance(as.numeric(data[PointsGj[m], ])  ,   centerGj)
    }
    EC[j] =mean(dm)

  }


  #Index
  Q_clusters=c(1:length(FHW_output$grouping))
  index_i=rep(0, length(FHW_output$grouping))

  for (i in 1:length(FHW_output$grouping)){
    vectj=Q_clusters[-i]


    dem=rep(0, length(FHW_output$grouping)-1  )
    num=rep(0, length(FHW_output$grouping)-1  )
    total=rep(0, length(FHW_output$grouping)-1  )
    for (j in 1:(length(FHW_output$grouping)-1)){

      dem[j]=distance(FHW_output$centers[i,] ,   FHW_output$centers[vectj[j],])
      num[j]=EC[i]+EC[vectj[j]]
      total[j]=num[j]/dem[j]
    }
    index_i[i]=max(total)
  }

  DaviesBouldin_index=mean(index_i)
  return(DaviesBouldin_index)

}
