#' Flexibilization of the Hartigan and Wong algorithm
#'
#'
#' This function implements the Hartigan and Wong algorithm (Hartigan and Wong, 1979) without imposing the use of the euclidean distance and without imposing that the centers of the groups are calculated by averaging the points. This function allow the use of  other distances and different ways to calculate the centers of the groups.
#'
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param k Number. Number of groups into which we are going to group the different points.
#' @param centers_function Function. This function designs how the centers of the groups will be calculated. It must have as input \code{data} and \code{grouping} and as output a matrix that has the centers. This matrix will have as many rows as centers. With \code{grouping} we mean a list. The list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#' @param init_centers Function. This function designs how we are going to calculate the initial centers. The input must be the \code{data}, \code{distance} and \code{k} and the output must be a matrix where each row has the center of one group.
#' @param seed Number. Number to fix a seed and be able to reproduce your results.
#' @param ITER Number. Maximum number of iterations.
#'
#' @return Returns a list with:
#'
#' \itemize{
#'   \item centers: the information of the centers updated. Matrix with \code{dim(centers)[1]} centers of \code{dim(centers)[2]} dimensions.
#'   \item grouping: the information of the groups updated. List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#'    }
#'
#'
#'
#'
#' @examples
#'
#'set.seed(451)
#'data=rbind(matrix(runif(20,1,5), nrow = 2, ncol = 10),
#'          matrix(runif(20,20,30), nrow = 2, ncol = 10),
#'          matrix(runif(20,50,70), nrow = 2, ncol = 10))
#'k=3
#'seed=5
#'
#'Hartigan_and_Wong(data,
#'                  Euclideandistance,
#'                  k,
#'                  centers_function_mean,
#'                  init_centers_random,
#'                  seed=seed,
#'                  10)
#'
#'
#'@references  Hartigan, J. A., & Wong, M. A. (1979). Algorithm AS 136: A k-means clustering algorithm. Journal of the royal statistical society. series c (applied statistics), 28(1), 100-108.
#'
#' @export





Hartigan_and_Wong<-function(data, distance,k,centers_function, init_centers, seed=NULL, ITER){

  data=as.matrix(data)
  colnames(data)<-NULL
  rownames(data)<-NULL

  #Just in case it is necessary
  if(!is.null(seed)){
    set.seed(seed)
  }

  #Step 0. Initialize centers
  centers=init_centers(data,distance,k,centers_function)


  #Step 1. We assign each point to its IC1. IC1 is the cluster with the closest center.
  #We calculate the distance between each row of the data matrix and the centers
  Dist_e_cent=matrix(0,dim(data)[1],dim(centers)[1])
  for (i in 1:(dim(data)[1])){
    for (j in 1:(dim(centers)[1])){
      Dist_e_cent[i,j]=distance(data[i,],centers[j,])
    }
  }

  #We obtain the IC1 and IC2 for each point
  Ic12=Dist_IC1_IC2(Dist_e_cent)
  Ic12_change=Ic12
  #We assign each point to IC1
  Group=Ic12[,1]
  grouping<-list()
  for(i in 1:(max(Group))){
    grouping[[i]]=which(Group==i)#list with the number of the rows of the data matrix that are in the group [[i]]
  }

  #If some group has no points
  if(any(sapply(grouping, function(x) length(x) == 0))==TRUE){

    EmptyGroups=which(sapply(grouping, function(x) length(x) == 0))

    np=rep(0,length(EmptyGroups) )
    old_group=rep(0,length(EmptyGroups) )
    tamOld=rep(0,length(EmptyGroups) )
    for (i in 1:length(EmptyGroups)){
      np[i]=which.min(Dist_e_cent[,EmptyGroups[i]])
      old_group[i]=encontrar_componente(grouping, np[i])
      tamOld[i]=length(grouping[[old_group[i]]])

    }

    if(all(!(old_group %in% EmptyGroups))==TRUE & any(tamOld==1)==FALSE){

      for (i in 1:length(EmptyGroups)){
        grouping[[old_group[i]]]=grouping[[old_group[i]]][grouping[[old_group[i]]]!=np[i]]
        grouping[[EmptyGroups[i]]]=np[i]
      }
    }else{
      leng_groups=sapply(grouping, length)
      PossibleGropus=which(leng_groups>length( EmptyGroups)+1)

      m=round(dim(data)[1]/k, 0)-1

      grouping[[1]]=c(1:m)

      for(i in 2:(k-1)){
        grouping[[i]]=c((m*(i-1)+1):(m*(i-1)+1+m-1))
      }

      grouping[[k]]=c((m*(k-1-1)+1+m-1+1):dim(data)[1])
    }


  }

  #Step 2. Update the clusters centers.
  centers=centers_function(data, grouping)

  #Step 3. Initially, all clusters belong to the live set.
  LIVE_SET_original=c(1:length(grouping))



  if(k==2){
    index=0

    #Step 4.
    Step4_output=Step4(data, centers, grouping, LIVE_SET_original, distance, centers_function, Ic12_change, index)
    centers=Step4_output$centers
    Ic12_change_after4=Step4_output$IC1andIC2
    grouping=Step4_output$grouping
    LIVE_SET_original=Step4_output$Live_set
    no_Change=Step4_output$no_Change
    index=Step4_output$index

    #Step 5.
    if (index==dim(data)[1]){
      HW_out=list(centers,grouping)
      names(HW_out)<-c("centers", "grouping")
      return(HW_out)
    }


    #Step 6
    Step6_output=Step6(data, centers, grouping, distance, centers_function, Ic12_change_after4,Ic12_change,index)
    centers=Step6_output$centers
    Ic12_change2=Step6_output$IC1andIC2
    grouping=Step6_output$grouping
    LIVE_SET_after_Step6=Step6_output$Live_set
    index=Step6_output$index


    HW_out=list(centers,grouping)
    names(HW_out)<-c("centers", "grouping")
    return(HW_out)
  }


  index=0
  for (mmm in 1:ITER){#If after doing Step4, Step5, Step6 the algorithm is not finished, we go back to Step4.

    #Step 4.
    Step4_output=Step4(data, centers, grouping, LIVE_SET_original, distance, centers_function, Ic12_change, index)
    centers=Step4_output$centers
    Ic12_change_after4=Step4_output$IC1andIC2
    grouping=Step4_output$grouping
    LIVE_SET_original=Step4_output$Live_set
    no_Change=Step4_output$no_Change
    index=Step4_output$index

    #Step 5.
    if (index==dim(data)[1]){
      HW_out=list(centers,grouping)
      names(HW_out)<-c("centers", "grouping")
      return(HW_out)
    }


    #Step 6
    Step6_output=Step6(data, centers, grouping, distance, centers_function, Ic12_change_after4,Ic12_change,index)
    centers=Step6_output$centers
    Ic12_change2=Step6_output$IC1andIC2
    grouping=Step6_output$grouping
    LIVE_SET_after_Step6=Step6_output$Live_set
    index=Step6_output$index


    #To return to Step4
    LIVE_SET_original=LIVE_SET_after_Step6
    Ic12_change=Ic12_change2


    #Stop criteria
    if (mmm==ITER){
      warning("The algorithm is stopped because the maximum number of iterations has been reached.")
      HW_out=list(centers,grouping)
      names(HW_out)<-c("centers", "grouping")
      return(HW_out)
    }
  }
}

