#' Hartigan and Wong algorithm
#'
#'
#' This function apply the \code{Hartigan_and_Wong} to different number of groups and calculates quality metrics as Silhouette.
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param centers_function Function. This function designs how the centers of the groups will be calculated. It must have as input \code{data} and \code{grouping} and as output a matrix that has the centers. This matrix will have as many rows as centers. With \code{grouping} we mean a list. The list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#' @param init_centers Function. This function designs how we are going to calculate the initial centers. The input must be the \code{data}, \code{distance} and \code{k} and the output must be a matrix where each row has the center of one group.
#' @param seed Number. Number to fix a seed and be able to reproduce your results.
#' @param ITER Number. Maximum number of iterations.
#' @param KK Number. Calculates the algorithm  for the number of groups 2,3,...,KK. Default \code{KK=10}.
#' @param index Character. If \code{index="Silhouette"} the function returns the results obtained with the number of groups (between 2 and KK) that maximize the Silhouette index. If \code{index="DaviesBouldin"} the function returns the results obtained with the number of groups (between 2 and KK) that minimize the Davies Bouldin index. If \code{index="Dunn"} the function returns the results obtained with the number of groups (between 2 and KK) that maximize the Dunn index. Default: "DaviesBouldin".
#' @param k Number. If k is not NULL the function returns the results obtained with  k groups.
#'
#'
#'
#' @return Returns a list with:
#'   \itemize{
#'   \item Number_of_groups: Number of groups took into account to cluster.
#'   \item Output_of_grouping: list with the centers and the clusters.
#'  \item Quality: vector with the Silhouette index, Davies Bouldin Index, the Dunn index, the Within Cluster Sum (WCS) and the time (in seconds) that the function \code{Hartigan_and_Wong} needs to be executed. The WCS is equal to the sum of the distance of each point to the center of its group.
#'    }
#'
#'
#'
#' @examples
#'
#'set.seed(451)
#'data=rbind(matrix(runif(20,1,5), nrow = 2, ncol = 10),
#'           matrix(runif(20,20,30), nrow = 2, ncol = 10),
#'           matrix(runif(20,50,70), nrow = 2, ncol = 10))
#'
#'
#'
#'RES=Hartigan_and_Wong_total(data,
#'                            RelativeDistance,
#'                            centers_function_RelativeDistance,
#'                            init_centers_random,
#'                            seed=10,
#'                            ITER=10,
#'                            KK=4,
#'                            index="DaviesBould",
#'                            k=NULL)
#'
#'@references  Hartigan, J. A., & Wong, M. A. (1979). Algorithm AS 136: A k-means clustering algorithm. Journal of the royal statistical society. series c (applied statistics), 28(1), 100-108.
#'
#' @export


Hartigan_and_Wong_total<-function(data, distance,centers_function, init_centers, seed=NULL, ITER, KK=10, index="DaviesBouldin", k=NULL){



  if(is.null(k)==FALSE){

    tim=system.time(Out<-Hartigan_and_Wong(data, distance,k,centers_function, init_centers, seed, ITER))["elapsed"]
    sil=mean(Silhouette(data,  Out, distance))
    dav=DaviesBouldinIndex(data, Out, distance)
    dunn=DunnIndex(data, Out, distance)
    WCSS=sum(ECDentroCluster3(data, Out, distance))

    Number_of_groups=k
    Output_of_grouping=Out
    Quality=c(sil, dav, dunn, WCSS, tim)
    names(Quality)<-c("Silhouette","DaviesBouldinIndex","DunnIndex" ,"WCS","Time")
    return(list(Number_of_groups=Number_of_groups, Output_of_grouping=Output_of_grouping, Quality= Quality))

  }else{

    pr<-list()
    Out<-list()
    sil=rep(0,KK)
    dav=rep(0,KK)
    dunn=rep(0,KK)
    WCSS=rep(0,KK)
    tim=rep(0,KK)
    for(k in 2:KK){

      tim[k]=system.time( Out[[k]]<-Hartigan_and_Wong(data, distance,k,centers_function, init_centers, seed, ITER))["elapsed"]
      sil[k]=mean(Silhouette(data,  Out[[k]], distance))
      dav[k]=DaviesBouldinIndex(data, Out[[k]], distance)
      dunn[k]=DunnIndex(data, Out[[k]], distance)
      WCSS[k]=sum(ECDentroCluster3(data, Out[[k]], distance))
    }


    if (index=="Silhouette"){
      Number_of_groups=which.max(sil)
      Output_of_grouping=Out[[Number_of_groups]]
      Quality=c(sil[Number_of_groups], dav[Number_of_groups], dunn[Number_of_groups],WCSS[Number_of_groups], tim[Number_of_groups])
      names(Quality)<-c("Silhouette","DaviesBouldinIndex","DunnIndex" ,"WCS","Time")
      return(list(Number_of_groups=Number_of_groups, Output_of_grouping=Output_of_grouping, Quality=Quality))
    }



    if (index=="DaviesBouldin"){
      Number_of_groups=which.min(dav[-1])+1
      Output_of_grouping=Out[[Number_of_groups]]
      Quality=c(sil[Number_of_groups], dav[Number_of_groups], dunn[Number_of_groups],WCSS[Number_of_groups], tim[Number_of_groups])
      names(Quality)<-c("Silhouette","DaviesBouldinIndex","DunnIndex","WCS","Time" )
      return(list(Number_of_groups=Number_of_groups, Output_of_grouping=Output_of_grouping, Quality= Quality))
    }


    if (index=="Dunn"){
      Number_of_groups=which.max(dunn)
      Output_of_grouping=Out[[Number_of_groups]]
      Quality=c(sil[Number_of_groups], dav[Number_of_groups], dunn[Number_of_groups],WCSS[Number_of_groups], tim[Number_of_groups])
      names(Quality)<-c("Silhouette","DaviesBouldinIndex","DunnIndex","WCS","Time" )
      return(list(Number_of_groups=Number_of_groups, Output_of_grouping=Output_of_grouping, Quality= Quality))
    }
  }






}
