% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/E_DHx_HmDm_HT.f.R
\name{E_DHx_HmDm_HT.f}
\alias{E_DHx_HmDm_HT.f}
\title{Estimate diameter and approximate confidence and prediction intervals}
\usage{
E_DHx_HmDm_HT.f(
  Hx,
  Hm,
  Dm,
  mHt,
  sHt = 0,
  par.lme,
  Rfn = list(fn = "sig2"),
  ...
)
}
\arguments{
\item{Hx}{Numeric vector of stem heights (m) along which to return the 
expected diameter.}

\item{Hm}{Numeric vector of stem heights (m) along which diameter 
measurements were taken for calibration. Can be of length 1. Must be of same 
length as \code{Dm}.}

\item{Dm}{Numeric vector of diameter measurements (cm) taken for calibration.
Can be of length 1. Must be of same length as \code{Hm}.}

\item{mHt}{Scalar. Tree height (m).}

\item{sHt}{Scalar. Standard deviation of stem height. Can be 0 if height was 
measured without error.}

\item{par.lme}{List of taper model parameters obtained by 
\code{\link{TapeR_FIT_LME.f}}.}

\item{Rfn}{list with function name to provide estimated or assumed residual 
variances for the given measurements, optionally parameters for such functions}

\item{...}{not currently used}
}
\value{
a list holding nine elements:
\itemize{
 \item{DHx: }{Numeric vector of diameters (cm) (expected value) along the 
 heights given by \code{Hx}.}
 \item{Hx: }{Numeric vector of heights (m) along which to return the expected
  diameter.}
 \item{MSE_Mean: }{Mean squared error for the expected value of the diameter.}
 \item{CI_Mean: }{Confidence interval. Matrix of the 95\% conf. int. for the 
 expected value of the diameter (cm). First column: lower limit, second 
 column: mean, third column: upper limit.}
 \item{KOV_Mean: }{Variance-Covariance matrix for the expected value of the 
 diameter.}
 \item{MSE_Pred: }{Mean squared error for the prediction of the diameter.}
 \item{CI_Pred: }{Prediction interval. Matrix of the 95\% conf. int. for the 
 prediction of the diameter (cm). First column: lower limit, second column: 
 mean, third column: upper limit.}
 \item{KOV_Pred: }{Variance-Covariance matrix for the prediction of the 
 diameter.}
 \item{Rfn: }{Function applied for estimated or assumed residual variance.}
}
}
\description{
Calibrates a taper curve based on at least one diameter 
measurement and returns the expected diameters and approximate variances
}
\details{
calibrates the tree specific taper curve and calculates approximate
confidence intervals, which can be useful for plotting. Uncertainty resulting
from tariff height estimates if tree height was not measured is incorporated.
Using \code{Rfn} the taper curve can be forced through the measured 
diameters, c.f. \code{\link{resVar}}.
}
\examples{
# example data
data(DxHx.df)
# taper curve parameters based on all measured trees
data(SK.par.lme)

#select data of first tree
Idi <- (DxHx.df[,"Id"] == unique(DxHx.df$Id)[1])
(tree1 <- DxHx.df[Idi,])

## Predict the taper curve based on the diameter measurement in 2 m
## height and known height 
tc.tree1 <- E_DHx_HmDm_HT.f(Hx=1:tree1$Ht[1], 
                            Hm=tree1$Hx[3],
                            Dm=tree1$Dx[3], 
                            mHt = tree1$Ht[1], 
                            sHt = 0, 
                            par.lme = SK.par.lme)
#plot the predicted taper curve
plot(tc.tree1$Hx, tc.tree1$DHx, type="l", las=1)
#lower CI
lines(tc.tree1$Hx, tc.tree1$CI_Mean[,1], lty=2)
#upper CI
lines(tc.tree1$Hx, tc.tree1$CI_Mean[,3], lty=2)
#lower prediction interval
lines(tc.tree1$Hx, tc.tree1$CI_Pred[,1], lty=3)
#upper prediction interval
lines(tc.tree1$Hx, tc.tree1$CI_Pred[,3], lty=3)
#add measured diameter used for calibration
points(tree1$Hx[3], tree1$Dx[3], pch=3, col=2)
#add the observations
points(tree1$Hx, tree1$Dx)

## feature of forcing taper curve through measured diameters
i <- c(3, 6)
tc.tree1 <- E_DHx_HmDm_HT.f(Hx=seq(0, tree1$Ht[1], 0.1), 
                            Hm=tree1$Hx[i],
                            Dm=tree1$Dx[i], 
                            mHt = tree1$Ht[1], 
                            sHt = 0, 
                            par.lme = SK.par.lme,
                            Rfn=list(fn="sig2"))
tc.tree2 <- E_DHx_HmDm_HT.f(Hx=seq(0, tree1$Ht[1], 0.1), 
                            Hm=tree1$Hx[i],
                            Dm=tree1$Dx[i], 
                            mHt = tree1$Ht[1], 
                            sHt = 0, 
                            par.lme = SK.par.lme,
                            Rfn=list(fn="zero"))
# plot the predicted taper curve
plot(tc.tree1$Hx, tc.tree1$DHx, type="l", las=1)
# added taper curve through measurement
points(x=tc.tree2$Hx, y=tc.tree2$DHx, type="l", lty=2)
# closer window
plot(tc.tree1$Hx, tc.tree1$DHx, type="l", las=1, xlim=c(0, 8), ylim=c(24, 30))
# added taper curve through measurement
points(x=tc.tree2$Hx, y=tc.tree2$DHx, type="l", lty=2)
# add measured diameter used for calibration
points(tree1$Hx[i], tree1$Dx[i], pch=3, col=2)
# add the observations
points(tree1$Hx, tree1$Dx)

## apply yet another residual variance function
i <- c(1, 2, 3) # calibrating with 0.5, 1m and 2m, assuming no error in 0.5m
zrv <- tree1$Hx[1] / tree1$Ht[1] # assumed zero resiudal variance
# assumed residual variance per measurement
TapeR:::resVar(relH = tree1$Hx[i] / tree1$Ht[1], fn = "dlnorm", 
               sig2 = SK.par.lme$sig2_eps, par = list(zrv=zrv))
tc.tree3 <- E_DHx_HmDm_HT.f(Hx=seq(0, tree1$Ht[1], 0.1), 
                            Hm=tree1$Hx[i],
                            Dm=tree1$Dx[i], 
                            mHt = tree1$Ht[1], 
                            sHt = 0, 
                            par.lme = SK.par.lme,
                            Rfn=list(fn="dlnorm", par=list(zrv=zrv)))
plot(tc.tree1$Hx, tc.tree1$DHx, type="l", las=1, xlim=c(0, 4))
points(x=tc.tree3$Hx, y=tc.tree3$DHx, type="l", lty=2)
points(tree1$Hx[i], tree1$Dx[i], pch=3, col=2)
points(tree1$Hx, tree1$Dx)
}
\references{
Kublin, E., Breidenbach, J., Kaendler, G. (2013) A flexible stem 
taper and volume prediction method based on mixed-effects B-spline 
regression, Eur J For Res, 132:983-997.
}
\seealso{
\code{\link{TapeR_FIT_LME.f}}
}
\author{
Edgar Kublin and Christian Vonderach
}
