#' Create annual counts report
#'
#' 'create_report_annual_counts' generates a data frame of annual case
#' counts for each disease, with years as columns.
#'
#' @inheritParams create_report_ytd_counts
#'
#' @returns Dataframe of annual counts with one row per disease and one column
#' per year.
#' @export
#'
#' @importFrom stats aggregate
#'
#' @examples
#' data <- data.frame(
#'   disease = c("A", "A", "B"),
#'   year = c(2020, 2021, 2020),
#'   counts = c(5, 7, 8)
#' )
#' create_report_annual_counts(data, diseases = c("A", "B", "C"))
create_report_annual_counts <- function(data, diseases) {
  # - Aggregate annual counts by disease and year
  annual_counts <- stats::aggregate(counts ~ disease + year,
                            data = data,
                            FUN = sum)

  # - Reshape data to use years as columns and diseases as rows
  annual_counts <- reshape_annual_wide(annual_counts)

  # - Add missing diseases
  annual_counts <- standardize_report_diseases(annual_counts, diseases)

  # - Clear row names
  rownames(annual_counts) <- NULL

  annual_counts
}


#' Create monthly counts report
#'
#' 'create_report_monthly_counts' generates a data frame of monthly case
#' counts for each disease for a specific year, with months as columns.
#'
#' @inheritParams create_report_ytd_counts
#'
#' @returns Dataframe of monthly counts with one row per disease and one column
#' per month (Jan through Dec).
#' @export
#'
#' @examples
#' data <- data.frame(
#'   disease = c("A", "A", "B", "B"),
#'   year = c(2024, 2024, 2024, 2023),
#'   month = c(1, 2, 1, 4),
#'   counts = c(5, 7, 8, 9)
#' )
#' create_report_monthly_counts(data, diseases = c("A", "B", "C"), y = 2024)
create_report_monthly_counts <- function(data, diseases, y) {
  # - Get monthly counts by disease, year, and month
  month_counts <- get_month_counts(data)

  # - Extract counts for given year
  month_counts <- month_counts[month_counts$year == y, ]

  # - Remove year column (don't want to include in report)
  month_counts$year <- NULL

  # - Reshape data to use months as columns and diseases as rows
  month_counts <- reshape_monthly_wide(month_counts)

  # - Add missing diseases
  month_counts <- standardize_report_diseases(month_counts, diseases)

  # - Clear row names
  rownames(month_counts) <- NULL

  month_counts
}


#' Create monthly averages report
#'
#' 'create_report_monthly_avgs' generates a data frame of average monthly case
#' counts for each disease across all years in the input data.
#'
#' Uses the following config options:
#' - `rounding_decimals`
#'
#' @inheritParams create_report_ytd_counts
#'
#' @returns Dataframe of monthly averages with one row per disease and one column
#' per month (Jan through Dec).
#' @inherit create_report_ytd_counts seealso
#' @export
#'
#' @importFrom stats aggregate
#'
#' @examples
#' data <- data.frame(
#'   disease = c("A", "A", "B", "B"),
#'   year = c(2023, 2024, 2023, 2024),
#'   month = c(1, 1, 2, 2),
#'   counts = c(10, 20, 15, 25)
#' )
#' config <- list(rounding_decimals = 1)
#' create_report_monthly_avgs(data, c("A", "B", "C"), config)
create_report_monthly_avgs <- function(data, diseases, config) {
  # - Compute average counts for each month
  monthly_avgs <- stats::aggregate(counts ~ disease + month,
                                   data = data,
                                   FUN = sum)

  num_yrs <- length(get_yrs(data))

  monthly_avgs$counts <- round(monthly_avgs$counts / num_yrs,
                               digits = config$rounding_decimals)

  # - Reshape data to use months as columns and disease as rows
  monthly_avgs <- reshape_monthly_wide(monthly_avgs)

  # - Add missing diseases
  monthly_avgs <- standardize_report_diseases(monthly_avgs, diseases)

  # - Clear row names
  rownames(monthly_avgs) <- NULL

  monthly_avgs
}


#' Create monthly medians report
#'
#' 'create_report_monthly_medians' generates a data frame of median monthly case
#' counts for each disease across all years in the input data. This provides a
#' more robust central tendency measure compared to averages for skewed data.
#'
#' @inheritParams create_report_ytd_counts
#'
#' @returns Dataframe of monthly medians with one row per disease and one column
#' per month (Jan through Dec).
#' @export
#'
#' @importFrom stats aggregate median reshape
#'
#' @examples
#' data <- data.frame(
#'   disease = c("A", "A", "A", "B", "B", "B"),
#'   year = c(2022, 2023, 2024, 2022, 2023, 2024),
#'   month = c(1, 1, 1, 2, 2, 2),
#'   counts = c(10, 20, 30, 5, 15, 25)
#' )
#' create_report_monthly_medians(data, c("A", "B", "C"))
create_report_monthly_medians <- function(data, diseases) {
  # - Get the full range of years in the data
  all_years <- get_yrs(data)
  all_months <- 1:12

  # - Compute counts for each month, aggregating by disease/month/year
  monthly_meds <- get_month_counts(data)

  # - Create a complete grid of all disease/year/month combinations
  # This ensures missing combinations are filled with 0
  complete_grid <- expand.grid(
    disease = diseases,
    year = all_years,
    month = all_months,
    stringsAsFactors = FALSE
  )

  # - Merge with actual data, filling missing values with 0
  monthly_meds <- merge(complete_grid, monthly_meds,
                        by = c("disease", "year", "month"),
                        all.x = TRUE)
  monthly_meds$counts[is.na(monthly_meds$counts)] <- 0

  # - Compute median counts for each disease/month across all years
  monthly_meds <- stats::aggregate(counts ~ disease + month,
                                   data = monthly_meds,
                                   FUN = median)

  # - Reshape data to use months as columns and disease as rows
  monthly_meds <- stats::reshape(
    monthly_meds,
    direction = "wide",
    idvar = "disease",
    timevar = "month"
  )
  # - Update column names to more human-readable format
  colnames(monthly_meds) <- c("disease", month.abb[1:(ncol(monthly_meds) - 1)])

  # - Clear row names
  rownames(monthly_meds) <- NULL

  monthly_meds
}


#' Create year-to-date (YTD) counts report
#'
#' 'create_report_ytd_counts' generates a data frame of year-to-date counts
#' for each disease up to the given month, comparing the given year to the
#' average of other years.
#'
#' Uses the following config options:
#' - `current_population`
#' - `avg_5yr_population`
#' - `rounding_decimals`
#'
#' @param data Dataframe. Input data with columns:
#' - `disease` (character)
#' - `year` (integer)
#' - `month` (integer)
#' - `counts` (integer)
#' @param diseases Character vector. Diseases to include in the report
#' @param y Integer. Report year
#' @param m Integer. Report month (1-12)
#' @param config List. Report settings
#' @param as.rates Logical. If TRUE, returns rates per 100k instead of raw counts
#'
#' @returns Dataframe with one row per disease and columns for current YTD and
#' average YTD values (either counts or rates per 100k)
#' @seealso [epitraxr_config()] for config options
#' @export
#'
#' @importFrom stats aggregate
#'
#' @examples
#' data <- data.frame(
#'   disease = c("A", "A", "B", "B"),
#'   year = c(2024, 2023, 2024, 2023),
#'   month = c(1, 1, 2, 2),
#'   counts = c(10, 20, 15, 25)
#' )
#' config <- list(
#'   current_population = 100000,
#'   avg_5yr_population = 100000,
#'   rounding_decimals = 1
#' )
#' create_report_ytd_counts(data, c("A", "B", "C"), 2024, 2, config)
create_report_ytd_counts <- function(data, diseases, y, m, config, as.rates = FALSE) {

  # - Aggregate monthly counts by disease, year, and month
  month_counts <- get_month_counts(data)

  num_prev_yrs <- length(unique(data[data$year != y,]$year))

  # Compute current year-to-date (YTD) counts
  current_ytd <- month_counts[month_counts$year == y, ]
  current_ytd <- stats::aggregate(counts ~ disease, data = current_ytd, FUN = sum)
  current_ytd <- standardize_report_diseases(current_ytd, diseases)

  # Compute average YTD counts for the previous years
  avg_5yr_ytd <- with(month_counts, month_counts[year != y & month <= m, ])
  avg_5yr_ytd <- stats::aggregate(counts ~ disease, data = avg_5yr_ytd, FUN = sum)
  avg_5yr_ytd$counts <- round(avg_5yr_ytd$counts / num_prev_yrs,
                              config$rounding_decimals)
  avg_5yr_ytd <- standardize_report_diseases(avg_5yr_ytd, diseases)

  # Assemble YTD report
  ytd_report <- data.frame(disease = current_ytd$disease)

  if (as.rates) {
    # Convert counts to rates per 100k
    ytd_report <- cbind(
      ytd_report,
      data.frame(
        Current_YTD_Rate_per_100k = convert_counts_to_rate(
          current_ytd$counts,
          pop = config$current_population,
          digits = config$rounding_decimals),
        Avg_5yr_YTD_Rate_per_100k = convert_counts_to_rate(
          avg_5yr_ytd$counts,
          pop = config$avg_5yr_population,
          digits = config$rounding_decimals)
    ))
  } else {
    # Use raw counts
    ytd_report <- cbind(
      ytd_report,
      data.frame(
        Current_YTD_Counts = current_ytd$counts,
        Avg_5yr_YTD_Counts = avg_5yr_ytd$counts
    ))
  }

  ytd_report
}


#' Create year-to-date (YTD) medians report
#'
#' 'create_report_ytd_medians' generates a data frame of median year-to-date
#' counts for each disease up to the given month (months `1:m`) across all years in the data.
#' This provides a robust central tendency measure for YTD values.
#'
#' @inheritParams create_report_ytd_counts
#'
#' @returns Dataframe with one row per disease and columns for disease name and
#' median YTD counts.
#' @export
#'
#' @importFrom stats aggregate median
#'
#' @examples
#' data <- data.frame(
#'   disease = c("A", "A", "A", "B", "B", "B"),
#'   year = c(2022, 2023, 2024, 2022, 2023, 2024),
#'   month = c(1, 1, 2, 2, 2, 3),
#'   counts = c(10, 15, 20, 5, 8, 12)
#' )
#' create_report_ytd_medians(data, c("A", "B", "C"), 2)
create_report_ytd_medians <- function(data, diseases, m) {

  # - Get the full range of years in the data
  all_years <- get_yrs(data)
  all_months <- 1:m

  # - Get YTD month counts
  ytd_medians <- get_month_counts(data)
  ytd_medians <- with(ytd_medians, ytd_medians[month <= m, ])

  # - Create a complete grid of all disease/year/month combinations
  # This ensures missing combinations are filled with 0
  complete_grid <- expand.grid(
    disease = diseases,
    year = all_years,
    month = all_months,
    stringsAsFactors = FALSE
  )

  # - Merge with actual data, filling missing values with 0
  ytd_medians <- merge(complete_grid, ytd_medians,
                        by = c("disease", "year", "month"),
                        all.x = TRUE)
  ytd_medians$counts[is.na(ytd_medians$counts)] <- 0

  # Sum counts by disease and year
  ytd_medians <- stats::aggregate(counts ~ disease + year,
                                   data = ytd_medians,
                                   FUN = sum)

  # Compute median counts for each disease/year
  ytd_medians <- stats::aggregate(counts ~ disease,
                                   data = ytd_medians,
                                   FUN = median)

  colnames(ytd_medians) <- c("disease", "median_counts")

  ytd_medians

}


#' Create grouped disease statistics report
#'
#' 'create_report_grouped_stats' generates a comprehensive report with current
#' and historical statistics for diseases organized by group. The report includes
#' monthly counts/rates, year-to-date counts, and trend analysis.
#'
#' Uses the following config options:
#' - `current_population`
#' - `avg_5yr_population`
#' - `rounding_decimals`
#' - `trend_threshold`
#'
#' @inheritParams create_report_ytd_counts
#' @param diseases Dataframe. Diseases to include in the report. Must have column
#' `EpiTrax_name` (character) with diseases to include. Optionally may have column
#' `Group_name` (character) to define disease groupings. If `Group_name` is
#' missing, all diseases will be grouped under "Uncategorized".
#'
#' @returns Dataframe with one row per disease containing:
#'   - Group: Disease group name
#'   - Disease: Disease name
#'   - Monthly counts and rates for current year/month
#'   - Historical monthly averages and medians
#'   - Year-to-date counts and historical averages and medians
#'   - YTD trend indicators
#' @seealso [create_report_monthly_counts()], [create_report_monthly_avgs()],
#' [create_report_monthly_medians()], [create_report_ytd_counts()],
#' [create_report_ytd_medians()] which this function uses and [epitraxr_config()]
#' for config options
#' @export
#'
#' @examples
#' data <- data.frame(
#'   disease = c("A", "A", "B", "B"),
#'   year = c(2023, 2024, 2023, 2024),
#'   month = c(1, 1, 2, 2),
#'   counts = c(10, 20, 15, 25)
#' )
#' diseases <- data.frame(
#'   EpiTrax_name = c("A", "B", "C"),
#'   Group_name = c("Group1", "Group1", "Group2")
#' )
#' config <- list(
#'   current_population = 100000,
#'   avg_5yr_population = 100000,
#'   rounding_decimals = 1,
#'   trend_threshold = 0.15
#' )
#' create_report_grouped_stats(data, diseases, 2024, 2, config)
create_report_grouped_stats <- function(data, diseases, y, m, config) {

  epitrax_disease_names <- diseases$EpiTrax_name

  # Check that disease groups were included
  if (is.null(diseases$Group_name)) {
    warning(
      "The parameter 'diseases' should contain a 'Group_name' column. ",
      "Since, no groups were provided, all diseases will be 'Uncategorized'."
    )
    diseases$Group_name <- rep("Uncategorized", length(epitrax_disease_names))
  }

  # Replace any NA values with "Uncategorized"
  diseases$Group_name[is.na(diseases$Group_name)] <- "Uncategorized"

  month_abb <- month.abb[m]
  month_name <- month.name[m]


  # Get current monthy/year counts
  grouped_r <- create_report_monthly_counts(
    data = data,
    diseases = epitrax_disease_names,
    y = y
  )
  grouped_r <- grouped_r[, c("disease", month_abb)]
  colnames(grouped_r) <- c("disease", "m_counts")


  # Get current monthy/year rate
  grouped_r$m_rates <- convert_counts_to_rate(
    counts = grouped_r$m_counts,
    pop = config$current_population,
    digits = config$rounding_decimals
  )


  # Get historical counts
  m_hist_avg_count <- create_report_monthly_avgs(
    data = data[data$year != y,],
    diseases = epitrax_disease_names,
    config = config
  )
  m_hist_avg_count <- m_hist_avg_count[, c("disease", month_abb)]
  colnames(m_hist_avg_count) <- c("disease", "m_hist_avg_count")
  grouped_r <- merge(grouped_r, m_hist_avg_count, by = "disease", all.x = TRUE)


  # Get historical median
  m_hist_median_count <- create_report_monthly_medians(
    data = data[data$year != y,],
    diseases = epitrax_disease_names
  )
  m_hist_median_count <- m_hist_median_count[, c("disease", month_abb)]
  colnames(m_hist_median_count) <- c("disease", "m_hist_median_count")
  grouped_r <- merge(grouped_r, m_hist_median_count, by = "disease", all.x = TRUE)

  # Get current and historical YTD counts
  y_ytd_stats <- create_report_ytd_counts(
    data = data,
    diseases = epitrax_disease_names,
    y = y,
    m = m,
    config = config,
    as.rates = FALSE
  )
  colnames(y_ytd_stats) <- c("disease", "y_YTD_count", "hist_y_ytd_avg_count")
  grouped_r <- merge(grouped_r, y_ytd_stats, by = "disease", all.x = TRUE)

  # Get historical YTD median
  y_ytd_medians <- create_report_ytd_medians(
    data = data[data$year != y,],
    diseases = epitrax_disease_names,
    m = m
  )
  colnames(y_ytd_medians) <- c("disease", "hist_y_ytd_median_count")
  grouped_r <- merge(grouped_r, y_ytd_medians, by = "disease", all.x = TRUE)

  # Get trend for YTD counts
  grouped_r$y_ytd_trend <- compute_trend(
    current = grouped_r$y_YTD_count,
    historical = grouped_r$hist_y_ytd_avg_count,
    threshold = config$trend_threshold
  )

  # Add disease groups to the report
  grouped_r <- merge(
    grouped_r,
    diseases[, c("EpiTrax_name", "Group_name")],
    by.x = "disease",
    by.y = "EpiTrax_name",
    all.x = TRUE
  )
  # - Rearrange columns to have Group_name first
  grouped_r <- grouped_r[,c(ncol(grouped_r),1:(ncol(grouped_r)-1))]

  # Update column names
  new_colnames <- c(
    "Group",
    "Disease",
    paste(month_name, y),
    paste(month_name, y, "Rate"),
    paste("Historical", month_name, "Avg"),
    paste("Historical", month_name, "Median"),
    paste(y, "YTD"),
    paste("Historical", y, "YTD Avg"),
    paste("Historical", y, "YTD Median"),
    "YTD Trend"
  )
  colnames(grouped_r) <- new_colnames

  # Final sort by Group and Disease
  grouped_r <- grouped_r[order(grouped_r$Group, grouped_r$Disease), ]

  grouped_r
}


#' Create combined monthly and year-to-date public report
#'
#' 'create_public_report_combined_month_ytd' creates a comprehensive public report
#' that combines monthly case data with year-to-date statistics for the given month
#' and year. This provides both current month context and cumulative year progress.
#'
#' Uses the following config options:
#' - `current_population`
#' - `avg_5yr_population`
#' - `rounding_decimals`
#' - `trend_threshold`
#'
#' @inheritParams create_report_ytd_counts
#' @param diseases Dataframe. Diseases to include in the report. Maps EpiTrax disease
#' names to public-facing versions. Must have columns:
#' - `EpiTrax_name` (character)
#' - `Public_name` (character)
#'
#' @returns List containing the report name and combined monthly/YTD report data
#' with columns for monthly cases/averages/trends and YTD statistics.
#' @seealso [create_public_report_month()], [create_report_ytd_counts()] which this
#' function uses and [epitraxr_config()] for config options
#' @export
#'
#' @examples
#' data_file <- system.file("sample_data/sample_epitrax_data.csv",
#'                          package = "epitraxr")
#' # Read in EpiTrax data
#' data <- read_epitrax_data(data_file)
#'
#' diseases <- data.frame(
#'   EpiTrax_name = c("Influenza", "COVID-19", "Measles", "Syphilis"),
#'   Public_name = c("Influenza", "COVID-19", "Measles", "Syphilis")
#' )
#' config_file <- system.file("tinytest/test_files/configs/good_config.yaml",
#'                           package = "epitraxr")
#' config <- get_report_config(config_file)
#' create_public_report_combined_month_ytd(
#'  data = data,
#'  diseases = diseases,
#'  y = 2024,
#'  m = 2,
#'  config = config
#' )
create_public_report_combined_month_ytd <- function(data, diseases, y, m, config) {

  # Modify the config for this function only because it returns rates, but we need counts
  m_report <- create_public_report_month(
    data = data,
    diseases = diseases,
    y = y,
    m = m,
    config = list(
      current_population = 100000,
      avg_5yr_population = 100000,
      rounding_decimals = config$rounding_decimals,
      trend_threshold = config$trend_threshold
    )
  )

  combined_r <- m_report$report

  # Create YTD report component
  ytd_report_cases <- create_report_ytd_counts(
    data = data,
    diseases = diseases$EpiTrax_name,
    y = y,
    m = m,
    config = config,
    as.rates = FALSE
  )
  ytd_report_rates <- create_report_ytd_counts(
    data = data,
    diseases = diseases$EpiTrax_name,
    y = y,
    m = m,
    config = config,
    as.rates = TRUE
  )
  ytd_report <- merge(ytd_report_cases, ytd_report_rates, by = "disease")

  # Convert disease names to public-facing versions
  ytd_report <- merge(ytd_report, diseases, by.x = "disease", by.y = "EpiTrax_name")
  ytd_report$disease <- ytd_report$Public_name
  ytd_report$Public_name <- NULL

  # Merge monthly and YTD reports
  combined_r <- merge(combined_r, ytd_report, by.x = "Disease", by.y = "disease")

  # Update column names
  new_colnames <- c(
    "Disease",
    paste(month.abb[m], "Cases"),
    paste(month.abb[m], "Average Cases"),
    "Trend",
    "YTD Cases",
    "YTD Average Cases",
    "YTD Rate per 100k",
    "YTD Average Rate per 100k"
  )

  colnames(combined_r) <- new_colnames

  # - Name and return report
  r_name <- paste0("public_report_combined_", month.abb[m], y)
  list(name = r_name, report = combined_r)
}


#' Create a monthly cross-section public report
#'
#' 'create_public_report_month' creates a public report for the given month.
#'
#' Uses the following config options:
#' - `current_population`
#' - `avg_5yr_population`
#' - `rounding_decimals`
#' - `trend_threshold`
#'
#' @inheritParams create_public_report_combined_month_ytd
#'
#' @returns List containing the report name and data.
#' @seealso  [get_month_counts()], [create_report_monthly_avgs()] which
#' this function uses and [epitraxr_config()] for config options
#' @export
#'
#' @importFrom stats aggregate
#'
#' @examples
#' data_file <- system.file("sample_data/sample_epitrax_data.csv",
#'                          package = "epitraxr")
#' # Read in EpiTrax data
#' data <- read_epitrax_data(data_file)
#'
#' diseases <- data.frame(
#'   EpiTrax_name = c("Influenza", "COVID-19", "Measles", "Syphilis"),
#'   Public_name = c("Influenza", "COVID-19", "Measles", "Syphilis")
#' )
#' config_file <- system.file("tinytest/test_files/configs/good_config.yaml",
#'                           package = "epitraxr")
#' config <- get_report_config(config_file)
#'
#' create_public_report_month(
#'  data = data,
#'  diseases = diseases,
#'  y = 2024,
#'  m = 1,
#'  config = config
#' )
create_public_report_month <- function(data, diseases, y, m, config) {

  # Get monthly counts by disease, year, and month
  cases <- get_month_counts(data)

  # Get monthly averages excluding the current year
  prev_yrs_data <- data[data$year != y,]
  avgs <- create_report_monthly_avgs(
      data = prev_yrs_data,
      diseases = diseases$EpiTrax_name,
      config = config
  )

  # Extract month name
  month_name <- month.abb[[m]]

  # - Extract counts for the given year and month
  m_counts <- with(cases, cases[year == y & month == m, c("disease", "counts")])

  # - Only take the rows with data in the final report
  m_counts <- m_counts[m_counts$disease %in% avgs$disease, ]

  # - Convert monthly average counts to rate per 100k
  m_rates <- convert_counts_to_rate(avgs[[month_name]],
                                    pop = config$avg_5yr_population,
                                    digits = config$rounding_decimals)

  # - Create the report data frame initializing the Rate_per_100k column to 0
  m_report <- data.frame(
    Disease = avgs$disease,
    Rate_per_100k = 0,
    Avg_5yr_Rate = m_rates
  )

  # - Update the Rate_per_100k column with values from m_counts
  for (i in 1:length(m_counts$disease)) {
    d <- m_counts$disease[i]
    rate <- convert_counts_to_rate(m_counts$counts[i],
                                   pop = config$current_population,
                                   digits = config$rounding_decimals)
    m_report[m_report$Disease == d, ]$Rate_per_100k <- rate
  }

  # - Convert disease names to public-facing versions
  m_report <- merge(m_report, diseases,
                    by.x = "Disease", by.y = "EpiTrax_name",
                    all.x = TRUE, all.y = FALSE)
  m_report$Disease <- m_report$Public_name
  m_report$Public_name <- NULL
  m_report <- m_report[order(m_report$Disease),]

  # - Combine diseases with same public name (if any)
  m_report <- stats::aggregate(m_report[ , -1],
                               by = list(Disease = m_report$Disease),
                               "sum")

  # - Add Trends column last
  m_report$Trend <- compute_trend(m_report$Rate_per_100k, m_report$Avg_5yr_Rate,
                              threshold = config$trend_threshold)

  # - Name and return report
  r_name <- paste0("public_report_", month_name, y)
  list(name = r_name, report = m_report)
}


#' Create a YTD public report
#'
#' 'create_public_report_ytd' creates a public report for YTD rates.
#'
#' Uses the following config options:
#' - `current_population`
#' - `avg_5yr_population`
#' - `rounding_decimals`
#' - `trend_threshold`
#'
#' @inheritParams create_public_report_combined_month_ytd
#'
#' @returns List containing the report name and data.
#' @seealso [create_report_ytd_counts()] which this function uses
#' and [epitraxr_config()] for config options
#' @export
#'
#' @importFrom stats aggregate
#'
#' @examples
#' data_file <- system.file("sample_data/sample_epitrax_data.csv",
#'                          package = "epitraxr")
#' # Read in EpiTrax data
#' data <- read_epitrax_data(data_file)
#'
#' diseases <- data.frame(
#'   EpiTrax_name = c("Influenza", "COVID-19", "Measles", "Syphilis"),
#'   Public_name = c("Influenza", "COVID-19", "Measles", "Syphilis")
#' )
#' config_file <- system.file("tinytest/test_files/configs/good_config.yaml",
#'                           package = "epitraxr")
#' config <- get_report_config(config_file)
#' create_public_report_ytd(
#'  data = data,
#'  diseases = diseases,
#'  y = 2024,
#'  m = 1,
#'  config = config
#' )
create_public_report_ytd <- function(data, diseases, y, m, config) {

  # Compute YTD rates
  ytd_rates <- create_report_ytd_counts(
      data = data,
      diseases = diseases$EpiTrax_name,
      y = y,
      m = m,
      config = config,
      as.rates = TRUE
  )

  # - Create the report data frame initializing the Rate_per_100k column to 0
  m_report <- data.frame(
    Disease = ytd_rates$disease,
    YTD_Rate_per_100k = ytd_rates$Current_YTD_Rate_per_100k,
    Avg_5yr_Rate = ytd_rates$Avg_5yr_YTD_Rate_per_100k
  )

  # - Convert disease names to public-facing versions
  m_report <- merge(m_report, diseases, by.x = "Disease", by.y = "EpiTrax_name")
  m_report$Disease <- m_report$Public_name
  m_report$Public_name <- NULL
  m_report <- m_report[order(m_report$Disease),]

  # - Combine diseases with same public name (if any)
  m_report <- stats::aggregate(m_report[ , -1],
                               by = list(Disease = m_report$Disease),
                               "sum")

  # - Add Trends column last
  m_report$Trend <- compute_trend(m_report$YTD_Rate_per_100k, m_report$Avg_5yr_Rate,
                              threshold = config$trend_threshold)

  # - Name and return report
  r_name <- "public_report_YTD"
  list(name = r_name, report = m_report)
}
