% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/probability_forest.R
\name{probability_forest}
\alias{probability_forest}
\title{Probability forest}
\usage{
probability_forest(
  X,
  Y,
  num.trees = 2000,
  sample.weights = NULL,
  clusters = NULL,
  equalize.cluster.weights = FALSE,
  sample.fraction = 0.5,
  mtry = min(ceiling(sqrt(ncol(X)) + 20), ncol(X)),
  min.node.size = 5,
  honesty = TRUE,
  honesty.fraction = 0.5,
  honesty.prune.leaves = TRUE,
  alpha = 0.05,
  imbalance.penalty = 0,
  ci.group.size = 2,
  compute.oob.predictions = TRUE,
  num.threads = NULL,
  seed = runif(1, 0, .Machine$integer.max)
)
}
\arguments{
\item{X}{The covariates.}

\item{Y}{The class label (must be a factor vector with no NAs).}

\item{num.trees}{Number of trees grown in the forest. Note: Getting accurate
confidence intervals generally requires more trees than
getting accurate predictions. Default is 2000.}

\item{sample.weights}{Weights given to an observation in estimation.
If NULL, each observation is given the same weight. Default is NULL.}

\item{clusters}{Vector of integers or factors specifying which cluster each observation corresponds to.
Default is NULL (ignored).}

\item{equalize.cluster.weights}{If FALSE, each unit is given the same weight (so that bigger
clusters get more weight). If TRUE, each cluster is given equal weight in the forest. In this case,
during training, each tree uses the same number of observations from each drawn cluster: If the
smallest cluster has K units, then when we sample a cluster during training, we only give a random
K elements of the cluster to the tree-growing procedure. When estimating average treatment effects,
each observation is given weight 1/cluster size, so that the total weight of each cluster is the
same.}

\item{sample.fraction}{Fraction of the data used to build each tree.
Note: If honesty = TRUE, these subsamples will
further be cut by a factor of honesty.fraction. Default is 0.5.}

\item{mtry}{Number of variables tried for each split. Default is
\eqn{\sqrt p + 20} where p is the number of variables.}

\item{min.node.size}{A target for the minimum number of observations in each tree leaf. Note that nodes
with size smaller than min.node.size can occur, as in the original randomForest package.
Default is 5.}

\item{honesty}{Whether to use honest splitting (i.e., sub-sample splitting). Default is TRUE.
For a detailed description of honesty, honesty.fraction, honesty.prune.leaves, and recommendations for
parameter tuning, see the grf algorithm reference.}

\item{honesty.fraction}{The fraction of data that will be used for determining splits if honesty = TRUE. Corresponds
to set J1 in the notation of the paper. Default is 0.5 (i.e. half of the data is used for
determining splits).}

\item{honesty.prune.leaves}{If TRUE, prunes the estimation sample tree such that no leaves
are empty. If FALSE, keep the same tree as determined in the splits sample (if an empty leave is encountered, that
tree is skipped and does not contribute to the estimate). Setting this to FALSE may improve performance on
small/marginally powered data, but requires more trees (note: tuning does not adjust the number of trees).
Only applies if honesty is enabled. Default is TRUE.}

\item{alpha}{A tuning parameter that controls the maximum imbalance of a split. Default is 0.05.}

\item{imbalance.penalty}{A tuning parameter that controls how harshly imbalanced splits are penalized. Default is 0.}

\item{ci.group.size}{The forest will grow ci.group.size trees on each subsample.
In order to provide confidence intervals, ci.group.size must
be at least 2. Default is 2.}

\item{compute.oob.predictions}{Whether OOB predictions on training set should be precomputed. Default is TRUE.}

\item{num.threads}{Number of threads used in training. By default, the number of threads is set
to the maximum hardware concurrency.}

\item{seed}{The seed of the C++ random number generator.}
}
\value{
A trained probability forest object.
}
\description{
Trains a probability forest that can be used to estimate
the conditional class probabilities P[Y = k | X = x]
}
\examples{
\donttest{
# Train a probability forest.
p <- 5
n <- 2000
X <- matrix(rnorm(n*p), n, p)
prob <- 1 / (1 + exp(-X[, 1] - X[, 2]))
Y <- as.factor(rbinom(n, 1, prob))
p.forest <- probability_forest(X, Y)

# Predict using the forest.
X.test <- matrix(0, 10, p)
X.test[, 1] <- seq(-1.5, 1.5, length.out = 10)
p.hat <- predict(p.forest, X.test, estimate.variance = TRUE)

# Plot the estimated success probabilities with 95 \% confidence bands.
prob.test <- 1 / (1 + exp(-X.test[, 1] - X.test[, 2]))
p.true <- cbind(`0` = 1 - prob.test, `1` = prob.test)
plot(X.test[, 1], p.true[, "1"], col = 'red', ylim = c(0, 1))
points(X.test[, 1], p.hat$predictions[, "1"], pch = 16)
lines(X.test[, 1], (p.hat$predictions + 2 * sqrt(p.hat$variance.estimates))[, "1"])
lines(X.test[, 1], (p.hat$predictions - 2 * sqrt(p.hat$variance.estimates))[, "1"])

# Predict on out-of-bag training samples.
p.hat <- predict(p.forest)
}

}
