% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hypothesize.R
\name{adjust_pval}
\alias{adjust_pval}
\title{Adjust P-Value for Multiple Testing}
\usage{
adjust_pval(x, method = "bonferroni", n = NULL)
}
\arguments{
\item{x}{A \code{hypothesis_test} object, or a list of such objects.}

\item{method}{Character. Adjustment method (see Details). Default is
\code{"bonferroni"}.}

\item{n}{Integer. Total number of tests in the family. If \code{x} is a list,
defaults to \code{length(x)}. For a single test, this must be specified.}
}
\value{
For a single test: a \code{hypothesis_test} object of subclass
\code{adjusted_test} with the adjusted p-value. For a list of tests: a list
of adjusted test objects.

The returned object contains:
\describe{
\item{stat}{Original test statistic (unchanged)}
\item{p.value}{Adjusted p-value}
\item{dof}{Original degrees of freedom (unchanged)}
\item{adjustment_method}{The method used}
\item{original_pval}{The unadjusted p-value}
\item{n_tests}{Number of tests in the family}
}
}
\description{
Applies a multiple testing correction to a hypothesis test or vector of
tests, returning adjusted test object(s).
}
\details{
When performing multiple hypothesis tests, the probability of at least one
false positive (Type I error) increases. Multiple testing corrections
adjust p-values to control error rates across the family of tests.

This function demonstrates the \strong{higher-order function} pattern: it takes
a hypothesis test as input and returns a transformed hypothesis test as
output. The adjusted test retains all original properties but with a
corrected p-value.
}
\section{Available Methods}{

The \code{method} parameter accepts any method supported by \code{\link[stats:p.adjust]{stats::p.adjust()}}:
\describe{
\item{\code{"bonferroni"}}{Multiplies p-values by \eqn{n}. Controls
family-wise error rate (FWER). Conservative.
}
\item{\code{"holm"}}{Step-down Bonferroni. Controls FWER. Less conservative
than Bonferroni while maintaining strong control.
}
\item{\code{"BH"} or \code{"fdr"}}{Benjamini-Hochberg procedure. Controls false
discovery rate (FDR). More powerful for large-scale testing.
}
\item{\code{"hochberg"}}{Step-up procedure. Controls FWER under independence.}
\item{\code{"hommel"}}{More powerful than Hochberg but computationally
intensive.
}
\item{\code{"BY"}}{Benjamini-Yekutieli. Controls FDR under arbitrary
dependence.
}
\item{\code{"none"}}{No adjustment (identity transformation).}
}
}

\section{Higher-Order Function Pattern}{

This function exemplifies transforming hypothesis tests:

\if{html}{\out{<div class="sourceCode">}}\preformatted{adjust_pval : hypothesis_test -> hypothesis_test
}\if{html}{\out{</div>}}

The output can be used with all standard generics (\code{pval()}, \code{test_stat()},
\code{is_significant_at()}, etc.) and can be further composed.
}

\examples{
# Single test adjustment (must specify n)
w <- wald_test(estimate = 2.0, se = 0.8)
pval(w)  # Original p-value

w_adj <- adjust_pval(w, method = "bonferroni", n = 10)
pval(w_adj)  # Adjusted (multiplied by 10, capped at 1)
w_adj$original_pval  # Can still access original

# Adjusting multiple tests at once
tests <- list(
  wald_test(estimate = 2.5, se = 0.8),
  wald_test(estimate = 1.2, se = 0.5),
  wald_test(estimate = 0.8, se = 0.9)
)

# BH (FDR) correction - n is inferred from list length
adjusted <- adjust_pval(tests, method = "BH")
sapply(adjusted, pval)  # Adjusted p-values

# Compare methods
sapply(tests, pval)  # Original
sapply(adjust_pval(tests, method = "bonferroni"), pval)  # Conservative
sapply(adjust_pval(tests, method = "BH"), pval)  # Less conservative

}
\seealso{
\code{\link[stats:p.adjust]{stats::p.adjust()}} for the underlying adjustment,
\code{\link[=fisher_combine]{fisher_combine()}} for combining (not adjusting) p-values
}
