% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est_mg.R
\name{est_mg}
\alias{est_mg}
\title{Multiple-group item calibration using MMLE-EM algorithm}
\usage{
est_mg(
  x = NULL,
  data,
  group.name = NULL,
  D = 1,
  model = NULL,
  cats = NULL,
  item.id = NULL,
  free.group = NULL,
  fix.a.1pl = FALSE,
  fix.a.gpcm = FALSE,
  fix.g = FALSE,
  a.val.1pl = 1,
  a.val.gpcm = 1,
  g.val = 0.2,
  use.aprior = FALSE,
  use.bprior = FALSE,
  use.gprior = TRUE,
  aprior = list(dist = "lnorm", params = c(0, 0.5)),
  bprior = list(dist = "norm", params = c(0, 1)),
  gprior = list(dist = "beta", params = c(5, 16)),
  missing = NA,
  Quadrature = c(49, 6),
  weights = NULL,
  group.mean = 0,
  group.var = 1,
  EmpHist = FALSE,
  use.startval = FALSE,
  Etol = 0.001,
  MaxE = 500,
  control = list(eval.max = 200, iter.max = 200),
  fipc = FALSE,
  fipc.method = "MEM",
  fix.loc = NULL,
  fix.id = NULL,
  se = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{x}{A list containing item metadata for all groups to be analyzed.
For example, if five groups are analyzed, the list should contain five
elements, each representing the item metadata for one group. The order of
the elements in the list must match the order of group names specified in
the \code{group.name} argument.

Each group's item metadata includes essential information for each item
(e.g., number of score categories, IRT model type, etc.) required for
calibration. See \code{\link[=est_irt]{est_irt()}} or \code{\link[=simdat]{simdat()}} for more
details about the item metadata.

When \code{use.startval = TRUE}, the item parameters specified in the metadata
will be used as starting values for parameter estimation. If \code{x = NULL},
both \code{model} and \code{cats} arguments must be specified. Note that when
\code{fipc = TRUE} to implement MG-FIPC, the \code{x} argument must be specified
and cannot be NULL. Default is \code{NULL}.}

\item{data}{A list containing item response matrices for all groups to be
analyzed. For example, if five groups are analyzed, the list should include
five elements, each representing the response data matrix for one group.
The elements in the list must be ordered to match the group names specified
in the \code{group.name} argument. Each matrix contains examinees' item responses
corresponding to the item metadata for that group. In each matrix, rows
represent examinees and columns represent items.}

\item{group.name}{A character vector indicating the names of the groups.
For example, if five groups are analyzed, use
\code{group.name = c("G1", "G2", "G3", "G4", "G5")}. Group names can be any
valid character strings.}

\item{D}{A scaling constant used in IRT models to make the logistic function
closely approximate the normal ogive function. A value of 1.7 is commonly
used for this purpose. Default is 1.}

\item{model}{A list containing character vectors specifying the IRT models
used to calibrate items across all groups. For example, if five groups are
analyzed, the list should contain five elements, each being a character
vector of IRT model names for one group. The elements in the list must be
ordered to match the group names specified in the \code{group.name} argument.

Available IRT models include:
\itemize{
\item \code{"1PLM"}, \code{"2PLM"}, \code{"3PLM"}, \code{"DRM"} for dichotomous items
\item \code{"GRM"}, \code{"GPCM"} for polytomous items
}

Here, \code{"GRM"} denotes the graded response model and \code{"GPCM"} the
(generalized) partial credit model. Note that \code{"DRM"} serves as a general
label covering all three dichotomous IRT models.If a single model name is
provided in any element of the list, it will be recycled across all items
within that group.

This argument is used only when \code{x = NULL} and \code{fipc = FALSE}. Default
is \code{NULL}.}

\item{cats}{A list containing numeric vectors specifying the number of score
categories for items in each group. For example, if five groups are
analyzed, the list should contain five numeric vectors corresponding to
the five groups. The elements in the list must be ordered consistently with
the group names specified in the \code{group.name} argument.

If a single numeric value is specified in any element of the list, it will
be recycled across all items in the corresponding group. If \code{cats = NULL}
and all models specified in the \code{model} argument are dichotomous
(i.e., "1PLM", "2PLM", "3PLM", or "DRM"), the function assumes that all
items have two score categories across all groups.

This argument is used only when \code{x = NULL} and \code{fipc = FALSE}. Default
is \code{NULL}.}

\item{item.id}{A list containing character vectors of item IDs for each group
to be analyzed. For example, if five groups are analyzed, the list should
contain five character vectors corresponding to the five groups. The elements
in the list must be ordered consistently with the group names specified in
the \code{group.name} argument.

When \code{fipc = TRUE} and item IDs are provided via the \code{item.id} argument,
the item IDs in the \code{x} argument will be overridden. Default is \code{NULL}.}

\item{free.group}{A numeric or character vector indicating the groups for which
the scales (i.e., mean and standard deviation) of the latent ability distributions
are freely estimated. The scales of the remaining groups (those not specified in this
argument) are fixed using the values provided in the \code{group.mean} and \code{group.var}
arguments, or from the \code{weights} argument.

For example, suppose that five groups are analyzed with group names "G1", "G2",
"G3", "G4", and "G5". To freely estimate the scales for groups 2 through 5, set
\code{free.group = c(2, 3, 4, 5)} or \code{free.group = c("G2", "G3", "G4", "G5")}.
In this case, the first group ("G1") will have a fixed scale
(e.g., a mean of 0 and variance of 1 when \code{group.mean = 0}, \code{group.var = 1},
and \code{weights = NULL}).}

\item{fix.a.1pl}{Logical. If \code{TRUE}, the slope parameters of all 1PLM items
are fixed to \code{a.val.1pl}; otherwise, they are constrained to be equal and
estimated. Default is \code{FALSE}.}

\item{fix.a.gpcm}{Logical. If \code{TRUE}, GPCM items are calibrated as PCM with
slopes fixed to \code{a.val.gpcm}; otherwise, each item's slope is estimated.
Default is \code{FALSE}.}

\item{fix.g}{Logical. If \code{TRUE}, all 3PLM guessing parameters are fixed to
\code{g.val}; otherwise, each guessing parameter is estimated. Default is
\code{FALSE}.}

\item{a.val.1pl}{Numeric. Value to which the slope parameters of 1PLM items
are fixed when \code{fix.a.1pl = TRUE}. Default is 1.}

\item{a.val.gpcm}{Numeric. Value to which the slope parameters of GPCM items
are fixed when \code{fix.a.gpcm = TRUE}. Default is 1.}

\item{g.val}{Numeric. Value to which the guessing parameters of 3PLM items
are fixed when \code{fix.g = TRUE}. Default is 0.2.}

\item{use.aprior}{Logical. If \code{TRUE}, applies a prior distribution to all
item discrimination (slope) parameters during calibration. Default is
\code{FALSE}.}

\item{use.bprior}{Logical. If \code{TRUE}, applies a prior distribution to all
item difficulty (or threshold) parameters during calibration. Default is
\code{FALSE}.}

\item{use.gprior}{Logical. If \code{TRUE}, applies a prior distribution to all
3PLM guessing parameters during calibration. Default is \code{TRUE}.}

\item{aprior, bprior, gprior}{A list specifying the prior distribution for all
item discrimination (slope), difficulty (or threshold), guessing
parameters. Three distributions are supported: Beta, Log-normal, and
Normal. The list must have two elements:
\itemize{
\item \code{dist}: A character string, one of \code{"beta"}, \code{"lnorm"}, or \code{"norm"}.
\item \code{params}: A numeric vector of length two giving the distribution’s
parameters. For details on each parameterization, see \code{\link[stats:Beta]{stats::dbeta()}},
\code{\link[stats:Lognormal]{stats::dlnorm()}}, and \code{\link[stats:Normal]{stats::dnorm()}}.
}

Defaults are:
\itemize{
\item \code{aprior = list(dist = "lnorm", params = c(0.0, 0.5))}
\item \code{bprior = list(dist = "norm", params = c(0.0, 1.0))}
\item \code{gprior = list(dist = "beta", params = c(5, 16))}
}

for discrimination, difficulty, and guessing parameters, respectively.}

\item{missing}{A value indicating missing responses in the data set. Default
is \code{NA}.}

\item{Quadrature}{A numeric vector of length two:
\itemize{
\item first element: number of quadrature points
\item second element: symmetric bound (absolute value) for those points
For example, \code{c(49, 6)} specifies 49 evenly spaced points from –6 to 6.
These points are used in the E-step of the EM algorithm. Default is \code{c(49,   6)}.
}}

\item{weights}{A two-column matrix or data frame containing the quadrature
points (in the first column) and the corresponding weights (in the second
column) for the latent ability prior distribution. If not \code{NULL}, the
latent ability distributions for the groups not specified in the \code{free.group}
argument are fixed to match the scale defined by the provided quadrature points
and weights. The weights and points can be conveniently generated using
the function \code{\link[=gen.weight]{gen.weight()}}.

If \code{NULL}, a normal prior density is used instead, based on the
information provided in the \code{Quadrature}, \code{group.mean}, and \code{group.var}
arguments. Default is \code{NULL}.}

\item{group.mean}{A numeric value specifying the mean of the latent variable
prior distribution when \code{weights = NULL}. Default is 0. For groups not
specified in the \code{free.group} argument, their distribution means are fixed
to this value in order to resolve the indeterminacy of the item parameter
scale.}

\item{group.var}{A positive numeric value specifying the variance of the
latent variable prior distribution when \code{weights = NULL}. Default is 1.
For groups not specified in the \code{free.group} argument, their distribution
variances are fixed to this value in order to resolve the indeterminacy of
the item parameter scale.}

\item{EmpHist}{Logical. If \code{TRUE}, the empirical histograms of the latent
ability prior distributions across all groups are estimated simultaneously
with the item parameters using the approach proposed by Woods (2007).
Item calibration is then performed relative to the estimated empirical priors.}

\item{use.startval}{Logical. If \code{TRUE}, the item parameters provided in the
item metadata (i.e., the \code{x} argument) are used as starting values for item
parameter estimation. Otherwise, internally generated starting values are
used. Default is \code{FALSE}.}

\item{Etol}{A positive numeric value specifying the convergence criterion for
the E-step of the EM algorithm. Default is \code{1e-4}.}

\item{MaxE}{A positive integer specifying the maximum number of iterations
for the E-step in the EM algorithm. Default is \code{500}.}

\item{control}{A list of control parameters to be passed to the optimization
function \code{\link[stats:nlminb]{stats::nlminb()}}. These parameters control the M-step of the EM
algorithm. For example, the maximum number of iterations in each M-step can
be specified using \code{control = list(iter.max = 200)}. The default maximum
number of iterations per M-step is 200. See \code{\link[stats:nlminb]{stats::nlminb()}} for
additional control options.}

\item{fipc}{Logical. If \code{TRUE}, multiple-group fixed item parameter
calibration (MG-FIPC) is applied during item parameter estimation.
When \code{fipc = TRUE}, the information on which items are fixed
must be provided via either \code{fix.loc} or \code{fix.id}. See below for details.}

\item{fipc.method}{A character string specifying the FIPC method. Available
options are:
\itemize{
\item \code{"OEM"}: No Prior Weights Updating and One EM Cycle (NWU-OEM; Wainer &
Mislevy, 1990)
\item \code{"MEM"}: Multiple Prior Weights Updating and Multiple EM Cycles
(MWU-MEM; Kim, 2006) When \code{fipc.method = "OEM"}, the maximum number of
E-steps is automatically set to 1, regardless of the value specified in
\code{MaxE}.
}}

\item{fix.loc}{A list of positive integer vectors. Each internal vector specifies
the positions of the items to be fixed in the item metadata (i.e., \code{x})
for each group when MG-FIPC is implemented (i.e., \code{fipc = TRUE}).
The internal objects in the list must follow the same order as the group names
provided in the \code{group.name} argument.

For example, suppose three groups are analyzed. In the first group,
the 1st, 3rd, and 5th items are fixed; in the second group, the 2nd, 3rd,
4th, and 7th items are fixed; and in the third group, the 1st, 2nd, and
6th items are fixed. Then
\code{fix.loc = list(c(1, 3, 5), c(2, 3, 4, 7), c(1, 2, 6))}.
Note that if the \code{fix.id} argument is not NULL, the information in \code{fix.loc}
will be ignored. See below for details.}

\item{fix.id}{A vector of character strings specifying the IDs of items to be
fixed when MG-FIPC is implemented (i.e., \code{fipc = TRUE}).

For example, suppose that three groups are analyzed. In the first group,
three items with IDs G1I1, C1I1, and C1I2 are fixed. In the second group,
four items with IDs C1I1, C1I2, C2I1, and C2I2 are fixed. In the third
group, three items with IDs C2I1, C2I2, and G3I1 are fixed.

In this case, there are six unique items fixed across the groups—namely,
G1I1, C1I1, C1I2, C2I1, C2I2, and G3I1, because C1I1 and C1I2 appear in
both the first and second groups, while C2I1 and C2I2 appear in both the
second and third groups. Thus, you should specify
\code{fix.id = c("G1I1", "C1I1", "C1I2", "C2I1", "C2I2", "G3I1")}.
Note that if the \code{fix.id} argument is not NULL, the information provided in
\code{fix.loc} is ignored. See below for details.}

\item{se}{Logical. If \code{FALSE}, standard errors of the item parameter
estimates are not computed. Default is \code{TRUE}.}

\item{verbose}{Logical. If \code{FALSE}, all progress messages, including
information about the EM algorithm process, are suppressed. Default is
\code{TRUE}.}
}
\value{
This function returns an object of class \code{est_irt}. The returned
object contains the following components:

\item{estimates}{A list containing two internal elements: \code{overall} and \code{group}.
The \code{overall} element is a data frame with item parameter estimates and their
standard errors, computed from the combined data across all groups. This data
frame includes only the unique items across all groups. The \code{group} element is
a list of group-specific data frames, each containing item parameter estimates
and standard errors for that particular group.}

\item{par.est}{A list with the same structure as \code{estimates}, containing only
the item parameter estimates (excluding standard errors), formatted according to
the item metadata structure.}

\item{se.est}{A list with the same structure as \code{estimates}, but containing only
the standard errors of the item parameter estimates. Note that the standard
errors are calculated using the cross-product approximation method
(Meilijson, 1989).}

\item{pos.par}{A data frame indicating the position index of each estimated
item parameter. This index is based on the combined data set across all groups
(i.e., the first internal object of \code{estimates}). The position information is
useful for interpreting the variance-covariance matrix of item parameter
estimates.}

\item{covariance}{A variance-covariance matrix of the item parameter
estimates, based on the combined data set across all groups
(i.e., the first internal object of \code{estimates}).}

\item{loglikelihood}{A list containing two internal objects (i.e., overall
and group) of marginal log-likelihood values based on the observed data.
The structure of the list matches that of \code{estimates}. Specifically,
the \code{overall} component contains the total log-likelihood summed across
all unique items from all groups, while the \code{group} component provides
group-specific log-likelihood values.}

\item{aic}{A model fit statistic based on the Akaike Information Criterion (AIC),
calculated from the log-likelihood of all unique items.}

\item{bic}{A model fit statistic based on the Bayesian Information Criterion (BIC),
calculated from the log-likelihood of all unique items.}

\item{group.par}{A list containing summary statistics (i.e., mean, variance,
and standard deviation) of the latent variable prior distributions across
all groups.}

\item{weights}{A list of two-column data frames, where the first column
contains quadrature points and the second column contains the corresponding
weights of the (updated) latent variable prior distributions for each group.}

\item{posterior.dist}{A matrix of normalized posterior densities for all
response patterns at each quadrature point. Rows and columns represent
response patterns and quadrature points, respectively.}

\item{data}{A list containing two internal objects (i.e., overall and group)
representing the examinees' response data sets. The structure of this
list matches that of the \code{estimates} component.}

\item{scale.D}{The scaling factor used in the IRT model.}

\item{ncase}{A list containing two internal objects (i.e., overall and group)
representing the total number of response patterns. The structure of this
list matches that of the \code{estimates} component.}

\item{nitem}{A list containing two internal objects (i.e., overall and group)
representing the total number of items included in the response data.
The structure of this list matches that of the \code{estimates} component.}

\item{Etol}{The convergence criterion for the E-step of the EM algorithm.}

\item{MaxE}{The maximum number of E-steps allowed in the EM algorithm.}

\item{aprior}{A list describing the prior distribution used for discrimination
parameters.}

\item{bprior}{A list describing the prior distribution used for difficulty
parameters.}

\item{gprior}{A list describing the prior distribution used for guessing
parameters.}

\item{npar.est}{The total number of parameters estimated across all unique items.}

\item{niter}{The number of completed EM cycles.}

\item{maxpar.diff}{The maximum absolute change in parameter estimates at
convergence.}

\item{EMtime}{Time (in seconds) spent on EM cycles.}

\item{SEtime}{Time (in seconds) spent computing standard errors.}

\item{TotalTime}{Total computation time (in seconds).}

\item{test.1}{First-order test result indicating whether the gradient
sufficiently vanished for solution stability.}

\item{test.2}{Second-order test result indicating whether the information matrix
is positive definite, a necessary condition for identifying a local maximum.}

\item{var.note}{A note indicating whether the variance-covariance matrix
was successfully obtained from the information matrix.}

\item{fipc}{Logical. Indicates whether FIPC was used.}

\item{fipc.method}{The method used for FIPC.}

\item{fix.loc}{A list containing two internal objects (i.e., overall and group)
indicating the locations of fixed items when FIPC is applied. The structure
of the list matches that of the 'estimates' component.}

Note that you can easily extract components from the output using the
\code{\link[=getirt]{getirt()}} function.
}
\description{
This function performs multiple-group item calibration (Bock & Zimowski, 1997)
using marginal maximum likelihood estimation via the expectation-maximization
(MMLE-EM) algorithm (Bock & Aitkin, 1981). It also supports multiple-group
fixed item parameter calibration (MG-FIPC; e.g., Kim & Kolen, 2016), which
extends the single-group FIPC method (Kim, 2006) to multiple-group settings.
For dichotomous items, the function supports one-, two-, and three-parameter
logistic IRT models. For polytomous items, the graded response model (GRM)
and the (generalized) partial credit model (GPCM) are available.
}
\details{
Multiple-group (MG) item calibration (Bock & Zimowski, 1996)
provides a unified framework for handling testing scenarios involving
multiple groups, such as nonequivalent groups equating, vertical scaling,
and the identification of differential item functioning (DIF). In such
applications, examinees from different groups typically respond to either
the same test form or to different forms that share common (anchor) items.

The goal of MG item calibration is to estimate both item parameters and
latent ability distributions for all groups simultaneously (Bock & Zimowski, 1996).
The \pkg{irtQ} package implements MG calibration via the \code{\link[=est_mg]{est_mg()}} function,
which uses marginal maximum likelihood estimation through the
expectation-maximization (MMLE-EM) algorithm (Bock & Aitkin, 1981).
In addition, the function supports multiple-group fixed item parameter
calibration (MG-FIPC; e.g., Kim & Kolen, 2016), which allows the parameters
of specific items to be fixed across groups.

In MG IRT analyses, it is common for multiple groups' test forms to share
some common (anchor) items. By default, the \code{\link[=est_mg]{est_mg()}} function
automatically constrains items with identical item IDs across groups
to share the same parameter estimates.

Most of the features of the \code{\link[=est_mg]{est_mg()}} function are similar to those of the
\code{\link[=est_irt]{est_irt()}} function. The main difference is that several arguments in
\code{\link[=est_mg]{est_mg()}} accept list objects containing elements for each group to be
analyzed. These arguments include \code{x}, \code{data}, \code{model}, \code{cats},
\code{item.id}, \code{fix.loc} and \code{fix.id}.

Additionally, \code{\link[=est_mg]{est_mg()}} introduces two new arguments: \code{group.name}
and \code{free.group}. The \code{group.name} argument is required to assign a unique
identifier to each group. The order of the list elements provided in
\code{x}, \code{data}, \code{model}, \code{cats}, \code{item.id}, \code{fix.loc} and \code{fix.id}
must match the order of group names specified in the \code{group.name} argument.

The \code{free.group} argument is required to indicate which groups have their
latent ability distribution scales (i.e., mean and standard deviation)
freely estimated. When no item parameters are fixed (i.e., \code{fipc = FALSE}),
at least one group must have a fixed latent ability scale (e.g., mean = 0 and
variance = 1) among the multiple groups sharing common items, in order to
resolve the scale indeterminacy inherent in IRT estimation. By specifying
the groups in the \code{free.group} argument, the scales for those groups will be
freely estimated, while the scales for all other groups not included in
\code{free.group} will be fixed using the values provided in the \code{group.mean} and
\code{group.var} arguments or from the \code{weights} argument.

Situations requiring the implementation of MG-FIPC typically arise when
new latent ability scales from multiple-group (MG) test data need to be
linked to an established scale (e.g., that of an existing item bank).
In a single run of the MG-FIPC procedure, the parameters of non-fixed
(freed) items across multiple test forms, as well as the latent ability
distributions for multiple groups, can be estimated on the same scale
as the fixed items (Kim & Kolen, 2016).

For example, suppose that three different test forms—Form 1, Form 2, and
Form 3—are administered to three nonequivalent groups: Group1, Group2,
and Group3. Form 1 and Form 2 share 12 common items (C1I1 to C1I12),
while Form 2 and Form 3 share 10 common items (C2I1 to C2I10). There are
no common items between Form 1 and Form 3. Also, assume that all unique
items in Form 1 are from an existing item bank and have already been
calibrated on the item bank's scale.

In this case, the goal of MG-FIPC is to estimate the parameters of all
items across the three test forms—except the unique items in Form 1—
and the latent ability distributions of the three groups, all on the
same scale as the item bank. To achieve this, the unique items in Form 1
must be fixed during MG-FIPC to link the current MG test data to the
item bank scale.

The \code{\link[=est_mg]{est_mg()}} function can implement MG-FIPC by setting
\code{fipc = TRUE}. In this case, the information on which items to fix must be
provided through either the \code{fix.loc} or \code{fix.id} argument. When using
\code{fix.loc}, you must supply a list of item positions (locations)
to be fixed in each group’s test form. For example, suppose that the test
data from the three groups above are analyzed. In the first group,
the 1st, 3rd, and 5th items are fixed; in the second group, the 2nd, 3rd,
4th, and 7th items are fixed; and in the third group, the 1st, 2nd, and
6th items are fixed. In this case, you would specify:
\code{fix.loc = list(c(1, 3, 5), c(2, 3, 4, 7), c(1, 2, 6))}.

Alternatively, you can use \code{fix.id} to specify a character vector of item IDs
to be fixed across groups. In the first group, the items with IDs G1I1, C1I1,
and C1I2 are fixed; in the second group, the items with IDs C1I1, C1I2,
C2I1, and C2I2 are fixed; and in the third group, the items with IDs C2I1,
C2I2, and G3I1 are fixed. In this case, there are six unique items to be
fixed across all groups: G1I1, C1I1, C1I2, C2I1, C2I2, and G3I1. You would
then specify:
\code{fix.id = c("G1I1", "C1I1", "C1I2", "C2I1", "C2I2", "G3I1")}.

Note that when both \code{fix.loc} and \code{fix.id} are provided, the information
in \code{fix.id} takes precedence and overrides \code{fix.loc}.
}
\examples{
\donttest{
## ------------------------------------------------------------------------------
# 1. MG calibration using the simMG data
#  - Details:
#    (a) Constrain common items between groups to have
#        identical item parameters (i.e., items C1I1–C1I12 between
#        Groups 1 and 2, and items C2I1–C2I10 between Groups 2 and 3).
#    (b) Freely estimate the means and variances of the ability
#        distributions for all groups except the reference group,
#        where the mean and variance are fixed to 0 and 1, respectively.
## ------------------------------------------------------------------------------
# 1-(1). Freely estimate the means and variances of Groups 2 and 3
# Import the true item metadata for the three groups
x <- simMG$item.prm

# Extract model, score category, and item ID information
# from the item metadata for the three groups
model <- list(x$Group1$model, x$Group2$model, x$Group3$model)
cats <- list(x$Group1$cats, x$Group2$cats, x$Group3$cats)
item.id <- list(x$Group1$id, x$Group2$id, x$Group3$id)

# Import the simulated response data sets for the three groups
data <- simMG$res.dat

# Import the group names for the three groups
group.name <- simMG$group.name

# Specify Groups 2 and 3 as the free groups where the scale
# of the ability distributions will be freely estimated.
# Group 1 will serve as the reference group, where the scale
# of the ability distribution is fixed to the values specified
# via the 'group.mean' and 'group.var' arguments
free.group <- c(2, 3) # or use 'free.group <- group.name[2:3]'

# Estimate IRT parameters:
# As long as common items across groups share the same item IDs,
# their item parameters will be constrained to be equal across groups
# unless FIPC is implemented.
fit.1 <-
  est_mg(
    data = data, group.name = group.name, model = model,
    cats = cats, item.id = item.id, D = 1, free.group = free.group,
    use.gprior = TRUE, gprior = list(dist = "beta", params = c(5, 16)),
    group.mean = 0, group.var = 1, EmpHist = TRUE, Etol = 0.001, MaxE = 500
  )

# Summary of the estimation
summary(fit.1)

# Extract the item parameter estimates
getirt(fit.1, what = "par.est")

# Extract the standard error estimates
getirt(fit.1, what = "se.est")

# Extract the group-level parameter estimates (i.e., scale parameters)
getirt(fit.1, what = "group.par")

# Extract the posterior latent ability distributions for each group
getirt(fit.1, what = "weights")

# 1-(2). Alternatively, the same parameter estimation can be performed by
# inserting a list of item metadata for the groups into the 'x' argument.
# If the item metadata contains item parameters to be used as starting values,
# set 'use.startval = TRUE'.
# Also, specify the groups in which the ability distribution scales
# will be freely estimated using their group names.
free.group <- group.name[2:3]
fit.2 <-
  est_mg(
    x = x, data = data, group.name = group.name, D = 1,
    free.group = free.group, use.gprior = TRUE,
    gprior = list(dist = "beta", params = c(5, 16)),
    group.mean = 0, group.var = 1, EmpHist = TRUE, use.startval = TRUE,
    Etol = 0.001, MaxE = 500
  )

# Summary of the estimation
summary(fit.2)

## ------------------------------------------------------------------------------
# 2. MG calibration with FIPC using simMG data
#  - Details:
#    (a) Fix the parameters of the common items between the groups
#        (i.e., items C1I1–C1I12 between Groups 1 and 2, and
#        items C2I1–C2I10 between Groups 2 and 3)
#    (b) Freely estimate the means and variances of the ability
#        distributions for all three groups
## ------------------------------------------------------------------------------
# 2-(1). Freely estimate the means and variances for all three groups
# Set all three groups as free groups in which the scales
# of the ability distributions are to be freely estimated
free.group <- 1:3 # or use 'free.group <- group.name'

# Specify the locations of items to be fixed in each group's metadata
# For Group 1: C1I1–C1I12 are located in rows 1–10 and 49–50
# For Group 2: C1I1–C1I12 are in rows 1–12, and
#              C2I1–C2I10 are in rows 41–50
# For Group 3: C2I1–C2I10 are in rows 1–10
fix.loc <- list(
  c(1:10, 49:50),
  c(1:12, 41:50),
  c(1:10)
)

# Estimate IRT parameters using MG-FIPC:
# When FIPC is implemented, item metadata for all groups
# must be provided via the 'x' argument.
# For fixed items, their item parameters must be specified
# in the metadata. For non-fixed items, any placeholder values
# can be used in the metadata.
# Also set fipc = TRUE and fipc.method = "MEM"
fit.3 <-
  est_mg(
    x = x, data = data, group.name = group.name, D = 1,
    free.group = free.group, use.gprior = TRUE,
    gprior = list(dist = "beta", params = c(5, 16)),
    EmpHist = TRUE, Etol = 0.001, MaxE = 500, fipc = TRUE,
    fipc.method = "MEM", fix.loc = fix.loc
  )

# Summary of the estimation
summary(fit.3)

# Extract the item parameter estimates
getirt(fit.3, what = "par.est")

# Extract the standard error estimates
getirt(fit.3, what = "se.est")

# Extract the group parameter estimates (i.e., scale parameters)
getirt(fit.3, what = "group.par")

# Extract the posterior latent ability distributions of the groups
getirt(fit.3, what = "weights")

# 2-(2). Alternatively, MG-FIPC can be implemented by specifying the
# IDs of the items to be fixed using the 'fix.id' argument.
# Provide a character vector of fixed item IDs to 'fix.id'
fix.id <- c(paste0("C1I", 1:12), paste0("C2I", 1:10))
fit.4 <-
  est_mg(
    x = x, data = data, group.name = group.name, D = 1,
    free.group = free.group, use.gprior = TRUE,
    gprior = list(dist = "beta", params = c(5, 16)),
    EmpHist = TRUE, Etol = 0.001, MaxE = 500, fipc = TRUE,
    fipc.method = "MEM", fix.id = fix.id
  )

# Summary of the estimation
summary(fit.4)

## ------------------------------------------------------------------------------
# 3. MG calibration with FIPC using simMG data
#    (Estimate group parameters only)
#  - Details:
#    (a) Fix all item parameters across all three groups
#    (b) Freely estimate the means and variances of the ability
#        distributions for all three groups
## ------------------------------------------------------------------------------
# 3-(1). Freely estimate the means and variances for all three groups
# Set all three groups as free groups in which the scales
# of the ability distributions will be freely estimated
free.group <- 1:3 # or use 'free.group <- group.name'

# Specify the locations of all fixed items in each group's metadata
fix.loc <- list(1:50, 1:50, 1:38)

# Estimate group parameters only using MG-FIPC
fit.5 <-
  est_mg(
    x = x, data = data, group.name = group.name, D = 1,
    free.group = free.group, use.gprior = TRUE,
    gprior = list(dist = "beta", params = c(5, 16)),
    EmpHist = TRUE, Etol = 0.001, MaxE = 500, fipc = TRUE,
    fipc.method = "MEM", fix.loc = fix.loc
  )

# Summary of the estimation
summary(fit.5)

# Extract the group parameter estimates (i.e., scale parameters)
getirt(fit.5, what = "group.par")

## ------------------------------------------------------------------------------
# 4. MG calibration with FIPC using simMG data
#    (Fix only the unique items of Group 1)
#  - Details:
#    (a) Fix item parameters of the unique items in Group 1 only
#    (b) Constrain the common items across groups to have
#        the same item parameters (i.e., C1I1–C1I12 between
#        Groups 1 and 2, and C2I1–C2I10 between Groups 2 and 3)
#    (c) Freely estimate the means and variances of the ability
#        distributions for all three groups
## ------------------------------------------------------------------------------
# 4-(1). Freely estimate the means and variances for all three groups
# Set all three groups as free groups in which the scales
# of the ability distributions will be freely estimated
free.group <- group.name # or use 'free.group <- 1:3'

# Specify the item IDs of the unique items in Group 1 to be fixed using
# the `fix.id` argument.
fix.id <- paste0("G1I", 1:38)

# Alternatively, use the 'fix.loc' argument as
# 'fix.loc = list(11:48, NULL, NULL)'

# Estimate IRT parameters using MG-FIPC
fit.6 <-
  est_mg(
    x = x, data = data, group.name = group.name, D = 1,
    free.group = free.group, use.gprior = TRUE,
    gprior = list(dist = "beta", params = c(5, 16)),
    EmpHist = TRUE, Etol = 0.001, MaxE = 500, fipc = TRUE,
    fipc.method = "MEM", fix.loc = NULL, fix.id = fix.id
  )

# Summary of the estimation
summary(fit.6)

# Extract the group parameter estimates (i.e., scale parameters)
getirt(fit.6, what = "group.par")

}


}
\references{
Bock, R. D., & Aitkin, M. (1981). Marginal maximum likelihood
estimation of item parameters: Application of an EM algorithm.
\emph{Psychometrika, 46}, 443-459.

Bock, R. D., & Zimowski, M. F. (1997). Multiple group IRT. In W. J.
van der Linden & R. K. Hambleton (Eds.), \emph{Handbook of modern item response theory}
(pp. 433-448). New York: Springer.

Kim, S. (2006). A comparative study of IRT fixed parameter calibration
methods. \emph{Journal of Educational Measurement, 43}(4), 355-381.

Kim, S., & Kolen, M. J. (2016). Multiple group IRT fixed-parameter estimation
for maintaining an established ability scale. \emph{Center for Advanced
Studies in Measurement and Assessment Report, 49.}

Meilijson, I. (1989). A fast improvement to the EM algorithm on its own
terms. \emph{Journal of the Royal Statistical Society: Series B
(Methodological), 51}, 127-138.

Woods, C. M. (2007). Empirical histograms in item response theory with
ordinal data. \emph{Educational and Psychological Measurement, 67}(1), 73-87.
}
\seealso{
\code{\link[=est_irt]{est_irt()}}, \code{\link[=shape_df]{shape_df()}}, \code{\link[=shape_df_fipc]{shape_df_fipc()}},
\code{\link[=getirt]{getirt()}}
}
\author{
Hwanggyu Lim \email{hglim83@gmail.com}
}
