#' Create Traceplots for MCMC Parameters
#'
#' @description
#' Creates traceplots for MCMC parameters (alpha, mu, sigma2) to assess convergence.
#' Users are responsible for setting up their preferred plot layout using par().
#'
#' @param mcmc_samples List containing MCMC samples for alpha, mu, and sigma2
#' @param param Parameter to plot: "alpha", "mu", "sigma2", or "all" (default: "all")
#' @param main Main title for the plot (optional)
#'
#' @return No return value, creates base R plots
#' 
#' @importFrom graphics abline
#' 
#' @examples
#' \donttest{
#' # Load the datasets
#' data(cancer_demo)
#' data(normal_demo)
#' rst <- mmcmcBayes(cancer_demo, normal_demo, 
#'                  stage = 1,max_stages = 2,num_splits = 5,
#'                  test = "BF", priors_cancer = NULL, priors_normal = NULL,
#'                  bf_thresholds = list(stage1 = 10, stage2 = 10.3, stage3 = 10.3),
#'                  return_mcmc = TRUE)
#' 
#' traceplot_asgn(rst$mcmc_samples$current_stage$cancer, param = "alpha", main = "Cancer Alpha")
#' }
#' @export
traceplot_asgn <- function(mcmc_samples, param = "all", main = NULL) {
  # Check if we have the expected structure
  required_params <- c("alpha", "mu", "sigma2")
  if (!all(required_params %in% names(mcmc_samples))) {
    stop("mcmc_samples must contain 'alpha', 'mu', and 'sigma2' components")
  }
  
  # Determine which parameters to plot
  if (identical(param, "all")) {
    params_to_plot <- required_params
  } else {
    params_to_plot <- intersect(param, required_params)
    if (length(params_to_plot) == 0) {
      stop("param must be one of: 'alpha', 'mu', 'sigma2', or 'all'")
    }
  }
  
  # Plot each requested parameter
  for (p in params_to_plot) {
    samples <- mcmc_samples[[p]]
    
    # Create title
    plot_title <- if (!is.null(main)) {
      paste(main, "-", p)
    } else {
      paste("Traceplot:", p)
    }
    
    # Create the plot
    plot(samples, type = "l",
         main = plot_title,
         xlab = "Iteration", ylab = p)
    
    # Add a horizontal line for the mean
    abline(h = mean(samples), col = "red", lty = 2, lwd = 1)
    
  }
}