% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lavaan_rerun.R
\name{lavaan_rerun}
\alias{lavaan_rerun}
\title{Rerun a 'lavaan' Analysis Using the Leaving-One-Out Approach}
\usage{
lavaan_rerun(
  fit,
  case_id = NULL,
  to_rerun,
  md_top,
  resid_md_top,
  allow_inadmissible = FALSE,
  skip_all_checks = FALSE,
  parallel = FALSE,
  ncores = NULL,
  makeCluster_args = list(spec = getOption("cl.cores", 2)),
  progress = TRUE,
  rerun_method = c("lavaan", "update")
)
}
\arguments{
\item{fit}{The output from \code{\link[lavaan:lavaan]{lavaan::lavaan()}} or its wrappers (e.g.,
\code{\link[lavaan:cfa]{lavaan::cfa()}} and \code{\link[lavaan:sem]{lavaan::sem()}}).}

\item{case_id}{If it is a character vector of length equals to the
number of cases (the number of rows in the data in \code{fit}), then it
is the vector of case identification values. If it is \code{NULL}, the
default, then \code{case.idx} used by \code{lavaan} functions will be used
as case identification values. The case identification values will
be used to name the list of \emph{n} output.}

\item{to_rerun}{The cases to be processed. If \code{case_id} is
specified, this should be a subset of \code{case_id}. If \code{case_id} is
not specified, then this should be a vector of integers indicating
the rows to te processed, as appeared in the data in \code{fit}.
\code{to_rerun} cannot be used together with \code{md_top} or
\code{resid_md_top.}}

\item{md_top}{The number of cases to be processed based on the
Mahalanobis distance computed on all observed variables used in
the model. The cases will be ranked from the largest to the
smallest distance, and the top \code{md_top} case(s) will be processed.
\code{md_top} cannot be used together with \code{to_rerun} or
\code{resid_md_top.}}

\item{resid_md_top}{The number of cases to be processed based on
the Mahalanobis distance computed from the residuals of outcome
variables. The cases will be ranked from the largest to the
smallest distance, and the top \code{resid_md_top} case(s) will be
processed. \code{resid_md_top} cannot be used together with \code{to_rerun}
or \code{md_top.}}

\item{allow_inadmissible}{If \code{TRUE}, accepts a fit object with
inadmissible results (i.e., \code{post.check} from
\code{\link[lavaan:lavInspect]{lavaan::lavInspect()}} is \code{FALSE}). Default is \code{FALSE}.}

\item{skip_all_checks}{If \code{TRUE}, skips all checks and allow
users to run this function on any object of \code{lavaan} class.
For users to experiment this and other functions on models
not officially supported. Default is \code{FALSE}.}

\item{parallel}{Whether parallel will be used. If \code{TRUE}, will use
functions in the \code{parallel} package to rerun the analysis.
Currently, only support \code{"snow"} type clusters using local CPU
cores. Default is \code{FALSE}.}

\item{ncores}{The number of CPU cores
to use if parallel processing is
requested. Default is \code{NULL}, and the
number of cores is determine by
\code{makeCluster_args}. If set to an
integer, this number will override
the setting (\code{spec}) in
\code{makeCluster_args}.}

\item{makeCluster_args}{A named list of arguments to be passed to
\code{\link[parallel:makeCluster]{parallel::makeCluster()}}. Default is \verb{list(spec = getOption("cl.cores", 2)))}. If only the number of cores need to
be specified, use \code{list(spec = x)}, where \code{x} is the number
of cores to use. Alternatively, set \code{ncores} and its
value will be used in \code{spec}.}

\item{progress}{If \code{TRUE}, the default,
progress will be displayed on screen.}

\item{rerun_method}{How fit will be rerun. Default is
\code{"lavaan"}. An alternative method is \code{"update"}. For
internal use. If \code{"lavaan"} returns an error, try setting
this argument to \code{"update"}.}
}
\value{
A \code{lavaan_rerun}-class object, which is a list with the following elements:
\itemize{
\item \code{rerun}: The \emph{n} \code{lavaan} output objects.
\item \code{fit}: The original output from \code{lavaan}.
\item \code{post_check}: A list of length equals to \emph{n}. Each analysis was
checked by \link[lavaan:lavInspect]{lavaan::lavTech}\verb{(x, "post.check")}, \code{x} being the
\code{lavaan} results. The results of this test are stored in this
list. If the value is \code{TRUE}, the estimation converged and the
solution is admissible. If not \code{TRUE}, it is a warning message
issued by \code{\link[lavaan:lavInspect]{lavaan::lavTech()}}.
\item \code{converged}: A vector of length equals to \emph{n}. Each analysis was
checked by \link[lavaan:lavInspect]{lavaan::lavTech}\verb{(x, "converged")}, \code{x} being the
\code{lavaan} results. The results of this test are stored in this
vector. If the value is \code{TRUE}, the estimation converged. If
not \code{TRUE}, then the estimation failed to converge if the corresponding
case is excluded.
\item \code{call}: The call to \code{\link[=lavaan_rerun]{lavaan_rerun()}}.
\item \code{selected}: A numeric vector of the row numbers of cases selected
in the analysis. Its length should be equal to the length of
\code{rerun}.
}
}
\description{
Reruns a \code{lavaan} analysis several
times, each time with one case removed.
}
\details{
\code{\link[=lavaan_rerun]{lavaan_rerun()}} gets an \code{\link[lavaan:lavaan]{lavaan::lavaan()}} output and
reruns the analysis \emph{n0} times, using the same arguments and
options in the output, \emph{n0} equals to the number of cases selected,
by default all cases in the analysis. In each
run, one case will be removed.

Optionally, users can rerun the analysis with only selected cases
removed. These cases can be specified by case IDs, by Mahalanobis
distance computed from all variables used in the model, or by
Mahalanobis distance computed from the residuals (observed score -
implied scores) of observed outcome variables. See the help on the
arguments \code{to_rerun}, \code{md_top}, and \code{resid_md_top}.

It is not recommended to use Mahalanobis distance computed from all
variables, especially for models with observed variables as
predictors (Pek & MacCallum, 2011). Cases that are extreme on
predictors may not be influential on the parameter estimates.
Nevertheless,
this distance is reported in some SEM programs and so this option
is provided.

Mahalanobis distance based on residuals are supported for models
with no latent factors. The implied scores are computed by
\code{\link[=implied_scores]{implied_scores()}}.

If the sample size is large, it is recommended to use parallel
processing. However, it is possible that parallel
processing will fail. If this is the case, try to use serial
processing, by simply removing the argument \code{parallel} or set it to
\code{FALSE}.

Many other functions in \link{semfindr} use the output from
\code{\link[=lavaan_rerun]{lavaan_rerun()}}. Instead of running the \emph{n} analyses every time, do
this step once and then users can compute whatever influence
statistics they want quickly.

If the analysis took a few minutes to run due to the large number
of cases or the long processing time in fitting the model, it is
recommended to save the output to an external file (e.g., by
\code{\link[base:readRDS]{base::saveRDS()}}).

Supports both single-group and multiple-group models.
(Support for multiple-group models available in 0.1.4.8 and later version).
}
\examples{
library(lavaan)
dat <- pa_dat
# For illustration, select only the first 50 cases
dat <- dat[1:50, ]
# The model
mod <-
"
m1 ~ iv1 + iv2
dv ~ m1
"
# Fit the model
fit <- lavaan::sem(mod, dat)
summary(fit)

# Fit the model n times. Each time with one case removed.
fit_rerun <- lavaan_rerun(fit, parallel = FALSE)

# Print the output for a brief description of the runs
fit_rerun

# Results excluding the first case
fitMeasures(fit_rerun$rerun[[1]], c("chisq", "cfi", "tli", "rmsea"))
# Results by manually excluding the first case
fit_01 <- lavaan::sem(mod, dat[-1, ])
fitMeasures(fit_01, c("chisq", "cfi", "tli", "rmsea"))

}
\author{
Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}.
}
